
-- Concatenates each of the supplied string arguments together.
local function concat(...)
	return table.concat({...})
end

-- Lua Pattern matching any English vowel, used by the _aOrAn() function below.
-- Note that 'u' is omitted, because it more commonly results in a 'y' sound
-- rather than a true 'u' sound when placed at the start of a word.
local _vowelSoundsPattern = '[aeioAEIO]'

-- Overly simplistic function to prefix nouns with the correct indefinite article
-- ('a' or 'an') so that they read correctly when printed. It's overly simplistic
-- in that it doesn't follow the silent 'h' rule etc, it just returns 'an' if the
-- noun starts with a vowel sound or 'a' in all other cases :) good enough...
local function aOrAn(noun)
	return (noun:sub(1, 1):match(_vowelSoundsPattern) and 'an' or 'a')
end

-- Returns 'was' if the supplied number is singular, or 'were' if it is plural or zero.
-- Used when printing strings like '1 argument was supplied' or '2 arguments were'.
local function wasOrWere(number)
	return number == 1 and 'was' or 'were'
end

-- Lookup table for the _indexToOrdinal() function below.
local _ordinals =
{
	'st',
	'nd',
	'rd',
	'th'
}

-- Used when reporting errors about a particular value supplied to an effect function,
-- so that we can append the correct ordinal suffix to the arg index we print (1st,
-- 2nd, 3rd, 4th, 5th etc).
local function indexToOrdinal(index)
	if index < #_ordinals then
		return index .. _ordinals[index]
	else
		return index .. _ordinals[#_ordinals]
	end
end

-- Converts the first letter of the supplied word to upper case.
local function uppercaseFirst(word)
	return word:sub(1, 1):upper() .. word:sub(2)
end

-- Public exports.
return
{
	concat = concat,
	aOrAn = aOrAn,
	wasOrWere = wasOrWere,
	indexToOrdinal = indexToOrdinal,
	uppercaseFirst = uppercaseFirst
}