-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local json = lazyRequire('cjson')

local function serializeExpression(expression)
	local expressionType = expression:getExpressionType()
	local serializedExpression = ""

	if (expressionType == "Animation") then
		serializedExpression = "Animation: { easing: " ..
				expression:getEasing() .. ", duration: " ..
				expression:getDuration() .. " }"
	elseif (expressionType == "Property") then
		serializedExpression = "Property"
	end

	return serializedExpression;
end

local function serialize(ruleId, ruleArray)

	local payload = {}

	-- Find the rule
	local rule = nil
	for k,r in pairs(ruleArray) do
		if (r.ruleId == ruleId) then
			rule = r;
		end
	end

	if rule == nil then
		payload[tostring(ruleId)] = nil
		return json.encode(payload)
	end

	local propertiesTable = {}
	for key,value in pairs(rule.styles:getPropertiesAsList()) do
		local val = value:getValue()
		if (type(val) == "table") then
			if (val.isExpression) then
				val = serializeExpression(val)
			else
				val = "Unknown table"
			end
		elseif (type(val) ~= "string") then
			val = tostring(val)
		end

		propertiesTable[value:getName()] = val
	end

	payload[tostring(ruleId)] =
	{
		file = rule._stylesheetPath,
		selector = rule.selector.rawString,
		properties = propertiesTable
	}

	local jsonPayload = json.encode(payload)

	-- Trim the leading and trailing {} object encapsulators, as we don't want to
	-- build a 'true' Json object, just a key/value pair
	jsonPayload = jsonPayload:sub(2, jsonPayload:len()-1)

	return jsonPayload
end

-- Public exports.
return
{
	serialize = serialize
}
