
--[[

	Symbol used for selecting nodes by one or more pseudo tags, e.g. ":loaded".

	Pseudo tags allow various aspects of the state of a node to be interrogated
	from the selector. This is useful for cases such as fading in images once
	they have loaded.

  ]]--

-- Character patterns.
local VALID_START_CHARACTERS  = '%:'
local VALID_MIDDLE_CHARACTERS = '[%a%d%-_]'

-- Prefix used by the Scene plugin to denote pseudo tags.
local PSEUDO_TAG_PREFIX = ':'

-- Set of valid pseudo tags used for catching typos etc in selectors.
local _validPseudoTags =
{
	hovered = true,
	focused = true,
	loaded = true,
	failed = true,
	reflowed = true
}

-- Symbol definition.
return
{
	-- Friendly name for this symbol type
	name = 'pseudoTag',

	-- Character to be used to begin this symbol in a selector. The pattern
	-- below matches just a ':'
	prefix = VALID_START_CHARACTERS,

	-- Function to be used to read this symbol from the raw selector string.
	-- This function will be called at the start of the symbol, and is expected
	-- to read until the end of the symbol, modifying the context as necessary.
	--
	-- Here we just keep reading until the end of the pseudo tag, and then add
	-- it to the current segment. The pattern matches any alphanumeric character,
	-- and also allows hyphens and underscores.
	reader = function(segment, parseContext)

		-- Skip past the ':' character, as we don't want it in the tag name
		parseContext:seek()

		-- Read the rest of the pseudo tag
		local pseudoTag = parseContext:readUntilMatchFails(VALID_MIDDLE_CHARACTERS)

		-- Check that we got a recognized pseudo tag
		if _validPseudoTags[pseudoTag] == nil then
			error('Unrecognized pseudo tag: ' .. pseudoTag)
		end

		-- Add the tag to the segment
		segment:addTag(PSEUDO_TAG_PREFIX .. pseudoTag)

	end,

	-- Function called when the number of occurrences of this symbol type in a
	-- given expression is being calculated. Supplied with a segment, the job
	-- of this function is just to return the number of symbols in the segment.
	counter = function(segment)

		-- Pseudo tags are placed into the tags array so are already taken
		-- care of by the tags counter - we can just return 0 here.
		return 0

	end
}
