
local _firstMatchIsLessSpecificThanSecondMatch

local MatchSorter = class()

-- Given a list of matches as returned by MatchingEngine:getMatchingRules(),
-- sorts the matches in ascending order by the specificity of the expression
-- that caused the match. For details as to how the specificity is calculated,
-- see SpecificityCalculator.calculate().
function MatchSorter:sortByMatchingExpressionSpecificity(matches)

	table.sort(matches, _firstMatchIsLessSpecificThanSecondMatch)

end

-- Sorting function used by sortByMatchingExpressionSpecificity(). Returns true
-- if the first match has lower specificity than the second, which results in
-- the matches array being sorted in ascending order of specificity.
--
-- Note that each specificity comprises a table made up of the counts of each
-- symbol type in the selector expression, e.g. { 3, 1 } for an expression that
-- has three tags and one node type. As such, we need to loop through each of
-- the counts, returning true if a count belonging to the first specificity table
-- is lower than a count belonging to the second specificity table at any point.
function _firstMatchIsLessSpecificThanSecondMatch(first, second)

	local firstSpecificity  = first.matchedExpression.specificity
	local secondSpecificity = second.matchedExpression.specificity

	for i = 1, #firstSpecificity do
		-- If the current value for the first specificity is lower than the
		-- one for the second specificity, we can break here and return true.
		if firstSpecificity[i] < secondSpecificity[i] then
			return true
		-- If the first is explicitly greater than the second (i.e. avoiding the
		-- case where the two are exactly equal), we can also bail here but this
		-- time return false.
		elseif firstSpecificity[i] > secondSpecificity[i] then
			return false
		end
	end

	-- If we got this far, it means the two specificities are exactly equal.
	-- In this case we return whether the rule was defined before the second
	-- rule.
	--
	-- The net result of this is that when rules have the same specificity, the
	-- order in which they were defined comes into play - rules which are defined
	-- later will take precedence over those which are defined earlier. This is
	-- happily the same behaviour as CSS uses, so works in our favour :)
	return first.rule.ruleId < second.rule.ruleId

end

return MatchSorter



