
local ffi = lazyRequire('ffi')
local Counters = plugins.style:lazyRequire('profiling.Counters')

local CandidateRuleGatherer = class()

-- Creates a rough list of candidate rules by leveraging the right-to-left
-- matching order of selectors and both the getRulesByRightmostTags() and
-- getRulesByRightHandSegmentNodeTypes() methods provided by RuleSet.
--
-- Essentially this just narrows down the list of rules that we have to run
-- the full matching algorithm over by creating an array of only those rules
-- where the rightmost segment can possibly return a match.
function CandidateRuleGatherer:getCandidateRules(node, ruleSet)

	-- Tracks the number of candidate rules that could be considered in the
	-- absolute worst case, i.e. if all rules were considered against all nodes.
	Counters.WORST_CASE_TOTAL_CANDIDATES:increment(ruleSet:getNumRules())

	local candidateRules = {}
	local addedRules = {}

	local nodePartitionHash = node:getPartitionAsHash()
	local ruleSetPartition = ruleSet:getPartitionByNameHash(nodePartitionHash)
	
	if ruleSetPartition ~= nil then
		self:_getCandidatesFromPartition(
				node, 
				ruleSetPartition, 
				candidateRules, 
				addedRules)
	end
	
	Counters.TOTAL_RULES_SELECTED_AS_CANDIDATES:increment(#candidateRules)

	return candidateRules

end

function CandidateRuleGatherer:_getCandidatesFromPartition(
		node, 
		partition, 
		candidateRules, 
		addedRules)

	-- First we get a list of all rule scopes that are relevant to the
	-- supplied node. This will include at least the global scope, but
	-- then usually some more specific viewScopes which will relate to the
	-- view or views that the current node is part of.
	local scopes = self:_getScopesRelevantToNode(node, partition)

	-- Create a table of node related info which is relevant to the candidate
	-- rule selection methods below, so that they don't have to retrieve it
	-- afresh each time they are called for each scope.
	local nodeInfo =
	{
		type = ffi.string(node:getType()),
		tags = node:getMatchabilityTagList(),
		numTags = node:getMatchabilityTagCountList()[0]
	}

	-- Then we iterate over each scope, extracting any relevant rules
	-- from each one.
	for i = 1, #scopes do
		local scope = scopes[i]
		
		-- First we use the scope:getRulesByRightHandSegmentNodeTypes() method
		-- to grab rules matching this node's node type in their right hand segment.
		self:_addRulesMatchingNodeType(nodeInfo, scope, candidateRules, addedRules)

		-- Then we look up each one in the scopes:getRulesByRightHandSegmentTags()
		-- table and build a full array of rules that we need to consider.
		self:_addRulesMatchingNodeTags(nodeInfo, scope, candidateRules, addedRules)
	end

end

local GLOBAL_HASH = crypto.djb2Hash('global')

-- Returns a list of all scopes which the current node should be matched
-- against. This starts with the global scope, and then includes any which
-- are related to views that the node sits within (this could be immediately
-- within, or at any level of depth).
function CandidateRuleGatherer:_getScopesRelevantToNode(node, partition)

	local globalScope = partition:getRulesByScope(GLOBAL_HASH)
	local scopes = { globalScope }
	local viewNamesArray = node:getMatchabilityViewNameList()
	local maxViewNames = 1000
	
	if globalScope ~= nil then
		Counters.CANDIDATES_FROM_GLOBAL_SCOPE
				:increment(#globalScope:getRules())
	end

	for i = 0, maxViewNames do
		local viewName = viewNamesArray[i]

		-- If we've reached the end of the view names array then bail.
		if viewName == 0xffffffff then
			break

		-- Otherwise, attempt to retrieve a scope matching the view name.
		else
			local viewScope = partition:getRulesByScope(viewName)

			if viewScope ~= nil then			
				Counters.CANDIDATES_FROM_VIEW_SCOPES
						:increment(#viewScope:getRules())
						
				table.insert(scopes, viewScope)
			end
		end
	end

	return scopes

end

-- Adds all rules to the supplied table which match the node type of the supplied
-- node in their rightmost symbol segment.
function CandidateRuleGatherer:_addRulesMatchingNodeType(
		nodeInfo, 
		scope, 
		candidateRules, 
		addedRules)

	local allRulesByNodeType = scope:getRulesByRightHandSegmentNodeTypes()
	local rulesWithThisNodeType = allRulesByNodeType[nodeInfo.type]

	if rulesWithThisNodeType then
		for i = 1, #rulesWithThisNodeType do
			local rule = rulesWithThisNodeType[i]

			-- If the rule specifies any tags then the _addRulesMatchingNodeTags
			-- method will pick it up and promote it for full analysis - in this
			-- case, on average, not including the node here results in fewer
			-- runs of the full matching algorithm as it's quite common to have
			-- a non-tagged node type dangling at the end of a selector segment.
			if (not rule.hasTagsInRightHandSegment) and (not addedRules[rule.ruleId]) then		
				Counters.RULES_INCLUDED_FROM_NODE_TYPE_BUCKETS:increment()
				
				addedRules[rule.ruleId] = true
				table.insert(candidateRules, rule)
			end
		end
	end

end

-- Adds all rules to the supplied table which mention any of the supplied
-- node's tags in their rightmost symbol segments.
function CandidateRuleGatherer:_addRulesMatchingNodeTags(
		nodeInfo, 
		scope, 
		candidateRules, 
		addedRules)

	local allRulesByTags = scope:getRulesByRightHandSegmentTags()

	-- Note that we have to iterate from 0, as the getTagsAsHashes() method
	-- returns a C array of uints which is 0 indexed.
	for i = 0, nodeInfo.numTags-1 do
		local thisTag = nodeInfo.tags[i]

		-- Bail if we've reached the end of the tags array.
		if thisTag == 0xffffffff then
			break
		end

		local rulesWithThisTag = allRulesByTags[thisTag]

		if rulesWithThisTag then
			for j = 1, #rulesWithThisTag do
				local rule = rulesWithThisTag[j]

				if (not addedRules[rule.ruleId]) then
					Counters.RULES_INCLUDED_FROM_RIGHTMOST_TAG_BUCKETS:increment()
				
					addedRules[rule.ruleId] = true
					table.insert(candidateRules, rule)
				end
			end
		end
	end

end

return CandidateRuleGatherer
