
local TypeRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.TypeRuntimeExpression')
local PropertyTypes = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')

-- Provides some validation functions for checking that values match a given type.
local PropertyTypeChecker = class()

-- Checks that the supplied value is both a number and also an integral number
local function _intChecker(value)
	if type(value) ~= 'number' then
		return type(value)
	elseif math.floor(value) ~= value then
		return 'float'
	else
		return nil
	end
end

-- Checks that the supplied value is an integral number and also non-negative
local function _uintChecker(value)
	if _intChecker(value) ~= nil then
		return _intChecker(value)
	elseif value < 0 then
		return 'int'
	else
		return nil
	end
end

-- Checks that the supplied value is of the supplied expected type
local function _genericTypeChecker(value, expectedType)
	if type(value) ~= expectedType then
		return type(value)
	else
		return nil
	end
end

-- Checks that the supplied value is a TypeRuntimeExpression of the expected type
local function _genericTypeRuntimeExpressionChecker(value, expectedFriendlyTypeName)
	if type(value) ~= 'table' then
		return type(value)
	elseif value.isExpression ~= true then
		return 'non-RuntimeExpression'
	elseif value:getExpressionType() ~= TypeRuntimeExpression.EXPRESSION_TYPE then
		return value:getExpressionType() .. 'RuntimeExpression'
	elseif value:getFriendlyTypeName() ~= expectedFriendlyTypeName then
		return value:getFriendlyTypeName()
	else
		return nil
	end
end

-- Map of property types to functions which check whether the supplied value matches
-- the expected type. If the value does not match the expected type, the function will
-- return the erroneous type that was supplied (e.g. the INT checking function will
-- return 'float' if a non-integral number is supplied, or 'string' if a string is
-- supplied). If the value does match the expected type, the function will return nil.
local _propertyTypesToCheckingFunctions =
{
	[PropertyTypes.INT] = function(value)
		return _intChecker(value)
	end,

	[PropertyTypes.UINT] = function(value)
		return _uintChecker(value)
	end,

	[PropertyTypes.FLOAT] = function(value)
		return _genericTypeChecker(value, 'number')
	end,

	[PropertyTypes.STRING] = function(value)
		return _genericTypeChecker(value, 'string')
	end,

	[PropertyTypes.BOOLEAN] = function(value)
		return _genericTypeChecker(value, 'boolean')
	end,

	[PropertyTypes.VEC2] = function(value)
		return _genericTypeRuntimeExpressionChecker(value, 'Vec2')
	end,

	[PropertyTypes.VEC3] = function(value)
		return _genericTypeRuntimeExpressionChecker(value, 'Vec3')
	end,

	[PropertyTypes.VEC4] = function(value)
		return _genericTypeRuntimeExpressionChecker(value, 'Vec4')
	end,

	[PropertyTypes.RECT] = function(value)
		return _genericTypeRuntimeExpressionChecker(value, 'Rect')
	end,
	
	[PropertyTypes.PIVOT_MODE] = function(value)
		return _genericTypeRuntimeExpressionChecker(value, 'PivotMode')
	end
}

-- Checks whether the supplied value matches the expected type. If it does, this
-- function will return nil. If not, it will return a context-aware string to inform
-- the user as to what mistake they've made - for example, supplying an int to a
-- uint will return the string 'int'; supplying a table to a float will return
-- the string 'table'.
function PropertyTypeChecker.checkForTypeMismatch(expectedType, value)
	return _propertyTypesToCheckingFunctions[expectedType](value)
end

return PropertyTypeChecker