-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local AbstractStyle = plugins.style:lazyRequire('rules.styles.definitions.AbstractStyle')
local PropertySetter = plugins.style:lazyRequire('rules.styles.definitions.PropertySetter')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')

--
-- Created for each key/value property pair specified in a list of styles, e.g.
-- for a rule like the following:
--
--    to '.foo' apply
--    {
--        x = 10,
--        y = 20
--    }
--
-- The styles list will contain two PropertyStyles: one with a name of x and a
-- value of 10, and a second with a name of y and value of 20.
--
local PropertyStyle = class(

	-- Parent class
	AbstractStyle,

	-- Constructor
	function(self, name, value, containingStyles, context)

		self:_init(containingStyles)

		self._name = name
		self._value = value
		self._context = context
		if AbstractRuntimeExpression.isExpression(value) then
			self._valueIsExpression = true
			self:_setupExpression(value)
		else
			self._valueIsExpression = false
		end
	end
)

function PropertyStyle:getName()
	return self._name
end

function PropertyStyle:getValue()
	return self._value
end

-- Applies this property style to the supplied node by using PropertyAccessDecorator
-- to transform the property name into its corresponding setter and then call the
-- setter method. Note that a static instance of PropertyAccessDecorator is used
-- so that the internal cache of setter names can be shared across all instances
-- of PropertyStyle.
function PropertyStyle:applyToNode(node, phase, isFirstAdditionOfEffect)

	local value

	-- If the value is an expression, evaluate it to ascertain the literal value.
	if self._valueIsExpression then
		value = self:_executeExpression(self._value, node, phase)
	-- Otherwise, we can just use the value directly - it's already a literal.
	else
		value = self:_evaluatePrimitive(self._value, node, phase)
	end
	-- Work out whether we should be animating out from a previously animated
	-- property value (i.e. if an animation has been applied to the property
	-- via an earlier style)
	local adjustedValue = self:_convertValueToAnimationTriggerIfAnimatingOut(node, value)
	-- Apply the value
	self:_setPropertyValue(node, adjustedValue, isFirstAdditionOfEffect)

	-- Store the value that was set
	self._context:getStyleRegistry():setLastAppliedValue(node, self, value)

end

function PropertyStyle:_setupExpression(expression)
	expression:setContext(self._context)
	expression:setAssociatedProperty(self)
end

function PropertyStyle:_executeExpression(expression, node, phase)
	local evaluator = self._context:getExpressionEvaluator()

	self:_setupExpression(expression)

	return evaluator:evaluate(expression, self, node, phase)
end

function PropertyStyle:_evaluatePrimitive(primitive, node, phase)
	if type(primitive) == 'table' then
		for key, value in pairs(primitive) do
			if AbstractRuntimeExpression.isExpression(value) then
				primitive[key] = self:_executeExpression(value, node, phase)
			end
		end
	end
	return primitive
end

-- If an earlier style that was applied to this property has added an animation
-- to it, then we need to animate 'out' from that style by converting the value
-- to an animation trigger with the same properties.
function PropertyStyle:_convertValueToAnimationTriggerIfAnimatingOut(node, value)

	-- If the value is already an animation trigger, this takes precedence over
	-- any animation that is already applied to the property.
	if AnimationTrigger.isAnimationTrigger(value) then
		return value

	-- Otherwise, check whether the previously applied value was an animation,
	-- and if so create a new animation based upon its easing, delay and duration.
	else
		local prevValue = self._context:getStyleRegistry():getLastAppliedValue(node, self)

		if AnimationTrigger.isAnimationTrigger(prevValue) then
			return prevValue:cloneWithNewToValue(value)
		else
			return value
		end
	end

end

function PropertyStyle:_setPropertyValue(node, value)

	-- If the evaluation returned an animation trigger, instruct it to start.
	if AnimationTrigger.isAnimationTrigger(value) then
		self._valueIsAnimation = true
		value:trigger()
	-- Otherwise, apply the value via the property setter.
	else
		self._propertySetter = self._propertySetter or PropertySetter.new(self._context)
		self._propertySetter:setPropertyOfNode(node, self._name, value)
	end

end

-- Does any work necessary to remove the style from the node - generally just
-- resetting properties to their default values, but also removing any bindings
-- that might have been created if the property value was a RuntimeExpression.
function PropertyStyle:removeFromNode(node)

	-- Clear up any bindings which were associated with this style, so that
	-- updates to the bindings will no longer affect properties of the node.
	if self._valueIsExpression then
		local bindingManager = self._context:getBindingManager()
		bindingManager:removeBindingsByNodeIdAndPropertyStyle(node:getId(), self)
	end

	-- If this style evaluates to an animation, stop the animation here.
	if self._valueIsAnimation then
		animation.clear(node, self._name)
	end

end

return PropertyStyle



