-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local djb2Hash = crypto.djb2Hash

local AbstractStyle = plugins.style:lazyRequire('rules.styles.definitions.AbstractStyle')
local EffectPropertyStyle = plugins.style:lazyRequire('rules.styles.definitions.EffectPropertyStyle')
local EffectLibrary = plugins.style:lazyRequire('effects.EffectLibrary')

local _getEffectPropertyName
local _getEffectPropertyIndex

--
-- Created for each handle/effect pair in a list of effects, e.g. for a rule
-- like the following:
--
--    to '.foo' apply
--    {
--        effects =
--        {
--            alpha = Alpha(1.0)
--            tint = Tint(Vec3(1.0, 0.0, 0.0))
--        }
--    }
--
-- The styles list will contain two EffectStyles: one with a handle of tint
-- and a value of Alpha(1.0), and the second with a name of tint and a value
-- of Tint(Vec3(1.0, 0.0, 0.0))
--
local EffectStyle = class(

	-- Parent class
	AbstractStyle,

	-- Constructor
	function(self, handle, value, containingStyles, context)

		self:_init(containingStyles)

		self._handle = handle
		self._context = context
		self._effectType = value.type
		self._effectProperties = {}

		-- Create an EffectPropertyStyle for each property of the effect.
		for uniformName, property in pairs(value.properties or {}) do

			local effectPropertyName = _getEffectPropertyName(uniformName, handle)
			local effectPropertyIndex = _getEffectPropertyIndex(effectPropertyName)

			self._effectProperties[uniformName] = EffectPropertyStyle.new(
				effectPropertyName,
				effectPropertyIndex,
				property.type,
				property.value,
				property.default,
				containingStyles,
				context
			)

		end

	end
)

-- Mirrors the behaviour of SceneNode::getEffectPropertyName() in order to
-- avoid having to call the native method from Lua every time an effect property
-- is applied to a node. Note that if the SceneNode::getEffectPropertyName()
-- implementation changes than that means this method does need to change too,
-- but this isn't likely to change often.
function _getEffectPropertyName(uniformName, handle)
	return uniformName .. '_' .. handle
end

-- Like _createEffectPropertyName, this mirrors the behaviour of the property
-- index creation method SceneNode::getEffectPropertyIndex(). This saves us
-- some round-trips to native and back, and is unlikely to change.
function _getEffectPropertyIndex(propertyName)
	return djb2Hash(propertyName)
end

function EffectStyle:getHandle()
	return self._handle
end

function EffectStyle:getEffectType()
	return self._effectType
end

function EffectStyle:getEffectProperties()
	return self._effectProperties
end

-- Map of effect functions used by the applyToNode() method
local _effectDefinitions = {}
EffectLibrary.loadEffects(_effectDefinitions)

-- Applies this effect style to the supplied node by using iterating over the
-- effect properties and calling setEffectProperty for each one in turn. If the
-- node does not already have the effect, it is added; if this effect style is
-- mapped to the special Remove() effect, it is removed.
function EffectStyle:applyToNode(node, phase, mutationTracker)

	-- If we've been supplied the special Remove() effect then we  need to
	-- attempt to remove any effect with the specified handle.
	if self._effectType == _effectDefinitions.Remove.TYPE then
		node:removeEffect(self._handle)

	-- If the node already has the effect, update it to match the style rule
	-- by setting each of its effect properties.
	--
	-- TODO: There's a possible optimisation to be made here - if the effect
	-- properties have not changed then there's no point setting them again.
	elseif node:hasEffect(self._handle) then
		self:_setEffectProperties(node, phase, false, mutationTracker)

	-- If the node doesn't already have the effect, add it and then set each
	-- of the effect properties.
	else
		node:addEffect(self._handle, self._effectType)
		self:_setEffectProperties(node, phase, true, mutationTracker)
	end

end

-- Iterates over each effect property belonging to the this effect, passing the
-- effect property values to the target node.
function EffectStyle:_setEffectProperties(node, phase, isFirstAdditionOfEffect, mutationTracker)

	local properties = self._effectProperties

	for i, property in pairs(properties) do
		property:applyToNode(node, phase, isFirstAdditionOfEffect)
		
		if mutationTracker ~= nil then
			mutationTracker:flagStyleApplicationForNode(node, property)
		end
	end

end

-- Removes this effect style to the supplied node.
function EffectStyle:removeFromNode(node)

	for i, property in pairs(self._effectProperties) do
		property:removeFromNode(node)
	end

end

return EffectStyle



