-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local PropertyStyle = plugins.style:lazyRequire('rules.styles.definitions.PropertyStyle')
local EffectStyle = plugins.style:lazyRequire('rules.styles.definitions.EffectStyle')
local ViewsStyle = plugins.style:lazyRequire('rules.styles.definitions.ViewsStyle')

--
-- Represents a parsed version of the raw styles list supplied when creating a
-- a rule. Essentially removes any 'special' values from the styles list which
-- require special treatment, and converts the remaining key/value pairs into the
-- list of node properties that make up the rest of the style.
--
local Styles = class(function(self, styles, context)

	self._propertiesByName = {}
	self._propertiesAsList = {}
	self._effects = nil
	self._viewProperties = nil
	self._context = context
	self._expressionEvaluator = self._context:getExpressionEvaluator()

	self:_parseStyles(styles or {})

	return self
end)

function Styles:getPropertiesByName()
	return self._propertiesByName
end

function Styles:getPropertiesAsList()
	return self._propertiesAsList
end

function Styles:getEffects()
	return self._effects
end

function Styles:getViewProperties()
	return self._viewProperties
end

function Styles:_parseStyles(styles)

	local specialValues = self:_removeSpecialValues(styles)

	-- Retrieve the effects list from the special values
	self:_buildEffectsList(specialValues.effects)

	-- Retrieve the view attributes list from the special values
	self:_buildViewPropertiesList(specialValues.viewProperties)

	-- Remaining styles are the the property key/value pairs
	self:_buildPropertiesList(styles)

end

-- List of values from the supplied style lists which require some form of special
-- treatment (see the _removeSpecialValues() method below).
--
-- Currently this is just the effects list, but is likely to include more keys
-- in future.
local _specialValueKeys =
{
	'effects',
	'viewProperties'
}

-- The majority of values in the styles list will just be properties of the node
-- that we need to set. However, there are some values which require special
-- treatment (the effects list is one example of this) and so must be removed
-- from the list of styles before continuing.
function Styles:_removeSpecialValues(styles)

	local specialValues = {}
	local key

	for i = 1, #_specialValueKeys do
		key = _specialValueKeys[i]
		specialValues[key] = styles[key]
		styles[key] = nil
	end

	return specialValues

end

-- Converts each of the input properties into a PropertyStyle object.
function Styles:_buildPropertiesList(rawProperties)

	for propertyName, propertyValue in pairs(rawProperties) do

		local property = PropertyStyle.new(
			propertyName,
			propertyValue,
			self,
			self._context)

		-- Insert into the properties list/array
		table.insert(self._propertiesAsList, property)

		-- Insert into the properties map
		self._propertiesByName[propertyName] = property

	end

end

-- Converts each of the input effects into an EffectStyle object.
function Styles:_buildEffectsList(rawEffects)

	if rawEffects ~= nil then
		self._effects = {}

		for effectHandle, effectValue in pairs(rawEffects) do

			local effect = EffectStyle.new(
				effectHandle,
				effectValue,
				self,
				self._context)

			self._effects[effectHandle] = effect

		end
	end

end

-- Converts each of the input view properties into an ViewsStyle object.
function Styles:_buildViewPropertiesList(rawProperties)
	if rawProperties ~= nil then
		if self._context:isViewsSupportEnabled() then
			self._viewProperties = {}

			for viewPropertyHandle, viewPropertyValue in pairs(rawProperties) do
				local viewProperty = ViewsStyle.new(
					viewPropertyHandle,
					viewPropertyValue,
					self,
					self._context)

				self._viewProperties[viewPropertyHandle] = viewProperty
			end
		else
			log.warn(
					'viewProperties will be ignored as the Views plugin has ' ..
					'not been loaded')
		end
	end
end

return Styles



