
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')

--
-- Maintains a map of node IDs to PropertyStyles and EffectStyles that are
-- currently being applied to them. Styles are contained in a map which allows
-- only one active style per property name or effect handle, so only the most
-- specific styles (i.e. based on the values returned by SpecificityCalculator)
-- will be listed against each node.
--
local StyleRegistry = class(function(self)

	self._nodeIdsToActiveStyles = {}
	self._nodeIdsToLastAppliedValues = {}

end)

-- Returns a map of all active styles for the supplied node.
--
-- If no active styles have yet been added, returns nil.
function StyleRegistry:getActiveStylesForNode(node)

	return self._nodeIdsToActiveStyles[node:getId()]

end

-- Sets the map of all active styles for the supplied node.
function StyleRegistry:setActiveStylesForNode(node, activeStyles)

	self._nodeIdsToActiveStyles[node:getId()] = activeStyles

end

-- Returns the value that was set the last time the supplied style was applied
-- to the supplied node.
function StyleRegistry:getLastAppliedValue(node, style)

	local lastAppliedValuesByStyle = self._nodeIdsToLastAppliedValues[node:getId()]

	if lastAppliedValuesByStyle ~= nil then
		return lastAppliedValuesByStyle[style:getName()]
	else
		return nil
	end

end

-- Stores the value that was set the last time the supplied style was applied
-- to the supplied node.
function StyleRegistry:setLastAppliedValue(node, style, value)

	local nodeId = node:getId()
	local styleName = style:getName()
	local lastAppliedValuesByStyle = self._nodeIdsToLastAppliedValues[nodeId]

	if lastAppliedValuesByStyle ~= nil then
		lastAppliedValuesByStyle[styleName] = value
	else
		self._nodeIdsToLastAppliedValues[nodeId] =
		{
			[styleName] = value
		}
	end

end

-- Removes any styles registered against the supplied node ID.
function StyleRegistry:removeStylesAndValuesForNodeId(nodeId)

	self._nodeIdsToActiveStyles[nodeId] = nil
	self._nodeIdsToLastAppliedValues[nodeId] = nil

end

return StyleRegistry