-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local TableUtils = plugins.style:lazyRequire('utils.TableUtils')
local Counters = plugins.style:lazyRequire('profiling.Counters')

--
-- Used by RuleApplier when working out what styles need to be applied to
-- a given node. Using the supplied StyleRegistry instance, calculates the
-- delta between styles currently active on a node (i.e. those applied in
-- earlier ticks) and styles which are to be applied on this tick. This data
-- then used by RuleApplier to work out which styles need applying and which
-- have already been applied.
--
local StyleListDeltaCalculator = class(function(self, styleRegistry)

	self._styleRegistry = styleRegistry

end)

--
-- Returns a table containing lists of the following style groups:
--
--     - added:      Styles which have been newly matched during the most recent
--                   matching pass (i.e. the property they have been applied to
--                   had no styling at all on the previous pass).
--
--     - removed:    Styles which matched during the previous matching pass but
--                   did not match this time (i.e. the property is now completely
--                   unstyled).
--
--     - updated:    Styles which have matched a different rule on this pass to
--                   the one they matched on the last pass (i.e. the style which
--                   previously matched will now have moved to the superseded list).
--
--     - superseded: Styles which still match but have now been occluded by more
--                   specific matches during the most recent matching pass.
--
function StyleListDeltaCalculator:calculateDelta(node, matches)

	Counters.DELTAS_CALCULATED:increment()
			
	local result   = self:_createResultContainer()
	local previous = self:_createShallowCopyOfPreviousActiveStyles(node)
	local current  = self:_determineCurrentActiveStyles(matches)

	self:_populateResultContainer(previous, current, result)
	self:_updateRegistry(node, current)

	return result

end

-- Creates an empty result container with added, removed and superseded tables.
function StyleListDeltaCalculator:_createResultContainer()
	return
	{
		added = {},
		removed = {},
		updated = {},
		superseded = {}
	}
end

-- Creates a copy of the existing active styles list just deep enough for its
-- values not to be affected when the styles list stored in the registry are
-- mutated.
function StyleListDeltaCalculator:_createShallowCopyOfPreviousActiveStyles(node)
	local existingStyles = self._styleRegistry:getActiveStylesForNode(node)

	if existingStyles ~= nil then
		return
		{
			properties     = TableUtils.shallowCopy(existingStyles.properties),
			effects        = TableUtils.shallowCopy(existingStyles.effects),
			viewProperties = TableUtils.shallowCopy(existingStyles.viewProperties)
		}
	else
		return
		{
			properties     = {},
			effects        = {},
			viewProperties = {}
		}
	end
end

-- Flattens the supplied list of matches down to a flat map of active styles
-- which should be applied to the node. Flattening is done with respect to the
-- specificity of the rules, so the dominant style for each property or effect
-- will be the one from the most specific matching rule.
function StyleListDeltaCalculator:_determineCurrentActiveStyles(matches)

	local activeProperties = {}
	local activeEffects = {}
	local activeViewProperties = {}

	for i = 1, #matches do
		local styles = matches[i].rule.styles
		local matchProperties     = styles:getPropertiesByName()
		local matchEffects        = styles:getEffects()
		local matchViewProperties = styles:getViewProperties()

		for key, property in pairs(matchProperties) do
			activeProperties[key] = property
		end

		-- Effects are optional, so may not be present
		if matchEffects ~= nil then
			for key, effect in pairs(matchEffects) do
				activeEffects[key] = effect
			end
		end

		-- ViewProperties are optional, so may not be present
		if matchViewProperties ~= nil then
			for key, viewProperty in pairs(matchViewProperties) do
				activeViewProperties[key] = viewProperty
			end
		end
	end

	return
	{
		properties = activeProperties,
		effects = activeEffects,
		viewProperties = activeViewProperties
	}

end

-- Iterates over the supplied previous and current styles list, populating
-- the added, removed and superseded tables in the resultsContainer with
-- the calculated deltas.
function StyleListDeltaCalculator:_populateResultContainer(previous, current, result)

	self:_calculateDelta(previous.properties,    current.properties,    result)
	self:_calculateDelta(previous.effects,       current.effects,       result)
	self:_calculateDelta(previous.viewProperties,current.viewProperties,result)

end

-- Called by _populateResultContainer() for each group of style types applied
-- to the node (i.e. the properties and effects groups), to do the work of
-- calculating the delta between the previous and current groups.
function StyleListDeltaCalculator:_calculateDelta(previous, current, result)

	for key, currentStyle in pairs(current) do
		local previousStyle = previous[key]
		local hasPrevious   = previousStyle ~= nil

		-- If there was no style with this key before, add it to the 'added' list.
		if (not hasPrevious) then
			table.insert(result.added, currentStyle)
		end

		-- If there was a style with this key before but the style has changed,
		-- add it to the 'updated' list. The style that it replaces then gets
		-- added to the 'superseded' list.
		if (hasPrevious) and (previousStyle ~= currentStyle) then
			table.insert(result.updated, currentStyle)
			table.insert(result.superseded, previousStyle)
		end
	end

	for key, previousStyle in pairs(previous) do
		local currentStyle = current[key]
		local hasCurrent   = currentStyle ~= nil

		-- If there is no style with this key now, add it to the 'removed' list.
		if not hasCurrent then
			table.insert(result.removed, previousStyle)
		end
	end

end

-- Stores all property styles and effect styles in the registry against the
-- supplied node.
function StyleListDeltaCalculator:_updateRegistry(node, activeStyles)

	self._styleRegistry:setActiveStylesForNode(node, activeStyles)

end

return StyleListDeltaCalculator