-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local IteratorsTable = plugins.style:lazyRequire("expressions.definitions.iterators.Iterators")
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local AbstractNodeKeyword = plugins.style:lazyRequire('rules.keywords.AbstractNodeKeyword')
local NodeKeywordFactory = plugins.style:lazyRequire('rules.keywords.NodeKeywordFactory')
local Styles = plugins.style:lazyRequire('rules.styles.Styles')

--
-- Used in tandem with RuleTableExporter to manage the transfer of rules from 
-- the rule definition Lua state to the rule application Lua state.
--
local RuleTableImporter = class(function(self, context)
	self._context = context
	self._expressionEvaluator = context:getExpressionEvaluator()
	self._convertedRulesCache = {}
end)

-- Note: For performance's sake this method will convert the incoming `rules`
-- argument from its table representation IN PLACE. If you want to preserve the
-- table then you'll have to do a deep copy on it before passing it to this method.
--
-- The `rulesContainer` argument is expected to be the output generated by 
-- RuleTableExporter:convertFromTableRepresentation().
function RuleTableImporter:convertFromTableRepresentation(rulesContainer)

	-- Hydrate any RuntimeExpression tables to turn them into actual instances
	self._convertedRulesCache = {}
	self:_convertFromTablesRecursively(rulesContainer.rules)

	for i, rule in ipairs(rulesContainer.rules) do
		-- Converts the rule's styles table into a proper Styles object, which
		-- in turn will split it into the effects list, properties list etc.
		rule.styles = Styles.new(rule.styles, self._context)
	end
	
	return rulesContainer
	
end

-- Checks whether any style values are RuntimeExpression tables, and if so
-- converts them to actual Expression instances via RuntimeExpression.fromTable().
function RuleTableImporter:_convertFromTablesRecursively(rules)

	for i, v in pairs(rules) do
		if type(rules[i]) == 'table' then
			if v.isIterator then
				rules[i] = IteratorsTable[v.iterator](unpack(v.args))
			elseif self._convertedRulesCache[v] ~= nil then
				-- Not only this avoids us re-recursing through a value
				-- that has already been converted, it also helps breaking
				-- potential infinite loops
				rules[i] = self._convertedRulesCache[v]
			else
				self:_convertFromTablesRecursively(rules[i])
			end
		end

		self._convertedRulesCache[v] = rules[i]
		if AbstractRuntimeExpression.isExpressionTable(v) then
			rules[i] = self._expressionEvaluator:createExpressionFromTable(v)
		elseif AbstractNodeKeyword.isKeywordTable(v) then
			rules[i] = NodeKeywordFactory.createKeywordFromTable(v)
		end
	end

end

return RuleTableImporter
