-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local djb2Hash = crypto.djb2Hash

local RuleScope = plugins.style:lazyRequire('rules.RuleScope')

--
-- Set of rules that have been mapped against a specific scene graph partition.
-- Rules in this set will only be considered for matching against nodes that 
-- match the provided partition name.
--
local RuleSetPartition = class(function(self, nameHash)

	self._nameHash = nameHash
	
	self:clear()
	
end)

function RuleSetPartition:clear()
	self._viewScopes = {}
	self._rulesAsArray = {}
	self._rulesAsSet = {}
end

function RuleSetPartition:getNameHash()
	return self._nameHash
end

function RuleSetPartition:addRules(rules)
	for i, rule in ipairs(rules) do
		if not self:hasRule(rule) then
			-- Add the rule to the relevant view scope, which will be the 
			-- global scope if the rule does not specify any.
			self:_addRuleToRelevantViewScope(rule)
			
			table.insert(self._rulesAsArray, rule)
			self._rulesAsSet[rule] = true
		end
	end
end

function RuleSetPartition:hasRule(rule)
	return self._rulesAsSet[rule] == true
end

-- Adds the supplied rule to the list of rules associated with its view scope.
--
-- If the scope is not yet known (i.e. this is the first rule that has been
-- encountered for this scope, a new scope will be created.
--
-- If the rule does not specify a scope, it will be added to the global set.
function RuleSetPartition:_addRuleToRelevantViewScope(rule)

	local viewScopeName   = rule.viewScope ~= nil and rule.viewScope or 'global'
	local viewScopeHash   = djb2Hash(viewScopeName)
	local viewScopeObject = self._viewScopes[viewScopeHash]

	if viewScopeObject == nil then
		viewScopeObject = RuleScope.new(viewScopeName)
		self._viewScopes[viewScopeHash] = viewScopeObject
	end

	viewScopeObject:addRule(rule)

end

function RuleSetPartition:getRules()
	return self._rulesAsArray
end

function RuleSetPartition:getRulesByScope(hash)
	return self._viewScopes[hash]
end

function RuleSetPartition:getNumRules()
	return #self._rulesAsArray
end

return RuleSetPartition



