
local bit = lazyRequire ('bit')
local TableUtils = plugins.style:lazyRequire('utils.TableUtils')

local _forEachRightSegmentInRule
local _forEachRightHandTagInRule
local _forEachRightHandNodeTypeInRule

--
-- List of rules which have been defined within a given scope (which may
-- be the global namespace, or may be related to a particular view).
--
-- Scopes provide a means to group together related rules so that rules
-- which aren't relevant to a particular node can be easily discarded when
-- running the matching algorithm.
--
local RuleScope = class(function(self, name)

	self._name = name
	self._rulesAsArray = {}
	self._rulesByRightHandSegmentTags = {}
	self._rulesByRightHandSegmentNodeTypes = {}
	
end)

RuleScope.NULL_TAG_MARKER = -1

function RuleScope:getName()
	return self._name
end

function RuleScope:getRules()
	return self._rulesAsArray
end

function RuleScope:getRulesByRightHandSegmentTags()
	return self._rulesByRightHandSegmentTags
end

function RuleScope:getRulesByRightHandSegmentNodeTypes()
	return self._rulesByRightHandSegmentNodeTypes
end

function RuleScope:getNumRules()
	return #self._rulesAsArray
end

-- Adds the supplied rule to this namespace.
function RuleScope:addRule(rule)

	table.insert(self._rulesAsArray, rule)

	-- Make the rules retrievable by different aspects of their right-hand
	-- selector segments, in order to facilitate faster matching at runtime.
	self:_addToRulesByRightHandSegmentTagsTable(rule)
	self:_addToRulesByRightHandSegmentNodeTypesTable(rule)

end

-- In order to provide faster matching at runtime, we store a lookup table of
-- each rule against the rightmost tags that it specifies - this is used by
-- MatchingEngine when building up a rough list of rules that should be considered
-- for matching before running the full matching algorithm against each one.
function RuleScope:_addToRulesByRightHandSegmentTagsTable(rule)

	local rulesByRightHandSegmentTags = self._rulesByRightHandSegmentTags

	_forEachRightHandTagInRule(rule, function(tag)

		local rulesForThisTag = rulesByRightHandSegmentTags[tag.hash]

		if not rulesForThisTag then
			rulesForThisTag = {}
			rulesByRightHandSegmentTags[tag.hash] = rulesForThisTag
		end

		if not TableUtils.arrayContainsValue(rulesForThisTag, rule) then
			table.insert(rulesForThisTag, rule)
		end

	end)

end

-- Does the equivalent of _addToRulesByRightHandSegmentTagsTable, but for the
-- node type specified in the selector rather than the tags.
function RuleScope:_addToRulesByRightHandSegmentNodeTypesTable(rule)

	local rulesByRightHandSegmentNodeTypes = self._rulesByRightHandSegmentNodeTypes

	_forEachRightHandNodeTypeInRule(rule, function(nodeType)

		local rulesForThisNodeType = rulesByRightHandSegmentNodeTypes[nodeType]

		if not rulesForThisNodeType then
			rulesForThisNodeType = {}
			rulesByRightHandSegmentNodeTypes[nodeType] = rulesForThisNodeType
		end

		if not TableUtils.arrayContainsValue(rulesForThisNodeType, rule) then
			table.insert(rulesForThisNodeType, rule)
		end

	end)

end

-- Calls the supplied function for each tag belonging to the last segment in
-- each selector of the supplied rule. For example...
--
-- Given:
--    ImageNode.foo bar.baz
--
-- Will call fn twice, first with 'bar' and then with 'baz', as the rightmost
-- segment is 'bar.baz'.
--
-- If the last segment in a selector doesn't have any tags, fn will be called
-- with an empty string as the tag.
function _forEachRightHandTagInRule(rule, fn)

	_forEachRightSegmentInRule(rule, function(rightHandSegment)
		local tags = rightHandSegment.tags

		if #tags > 0 then
			rule.hasTagsInRightHandSegment = true

			for k = 1, #tags do
				fn(tags[k])
			end
		else
			fn(
			{
				name = '',
				hash = RuleScope.NULL_TAG_MARKER 
			})
		end
	end)

end

-- Calls the supplied function with the node type belonging to the last segment
-- in each selector expression of the supplied rule. For example...
--
-- Given:
--    ContainerNode.foo ImageNode, ContainerNode.bar TextNode
--
-- Will call fn twice, first with 'ImageNode' and then with 'TextNode', as the
-- rightmost segments in each expression are 'ImageNode' and 'TextNode'.
--
-- If the last segment in a selector doesn't specify a node type, fn will be
-- called with an asterisk (i.e. the wildcard operator) as the node type.
function _forEachRightHandNodeTypeInRule(rule, fn)

	_forEachRightSegmentInRule(rule, function(rightHandSegment)
		fn(rightHandSegment.nodeType)
	end)

end

-- Calls the supplied function for each right hand segment in the supplied rule.
function _forEachRightSegmentInRule(rule, fn)

	local selector = rule.selector
	local expressions = selector.expressions

	for i = 1, #expressions do
		local expression = expressions[i]
		local rightHandSegment = expression.segments[#expression.segments]

		fn(rightHandSegment)
	end

end

return RuleScope



