-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local Selector = plugins.style:lazyRequire('selectors.Selector')
local SelectorExpression = plugins.style:lazyRequire('selectors.SelectorExpression')
local SelectorExpressionSegment = plugins.style:lazyRequire('selectors.SelectorExpressionSegment')
local ParsingEngine = plugins.style:lazyRequire('selectors.parsing.ParsingEngine')
local Rule = plugins.style:lazyRequire('rules.Rule')
local RuleTableExporter = plugins.style:lazyRequire('rules.RuleTableExporter')
local ExpressionEvaluator = plugins.style:lazyRequire('expressions.ExpressionEvaluator')

--
-- Provides the beginRule() and endRule() methods used at rule definition time
-- when the 'to' and 'apply' keywords are used in stylesheets. Essentially
-- maintains a tiny state machine in which there is one 'open rule' created
-- when the 'to' keyword is called, and then subsequently closed (i.e. becoming
-- a complete rule) when the 'apply' keyword is called.
--
local RuleDefiner = class(function(self, ruleFinalizer)

	self._parser = ParsingEngine.new()
	self._exporter = RuleTableExporter.new()
	self._finalizer = ruleFinalizer
	assert(type(self._finalizer.finalize) == "function",
			"The rule finalizer provided to RuleDefiner must expose a 'finalize' function")

	-- Path to the stylesheet currently being interpreted. This is used for
	-- tracking which stylesheet each rule came from. 
	--
	-- Note that if the rules are being loaded directly from a string rather 
	-- than a file (i.e. by just calling LuaState::doString() from C++), it's 
	-- valid for this to be null.
	self._currentStylesheetPath = nil
	
	self:clear()

end)

function RuleDefiner:clear()
	self._rules = {}
	self._partitionMappings = {}
end

function RuleDefiner:getRules()
	return self._rules
end

function RuleDefiner:getNumRules()
	return #self._rules
end

function RuleDefiner:getCurrentRule()
	return self._rules[#self._rules]
end

function RuleDefiner:setCurrentStylesheetPath(path)
	self._currentStylesheetPath = path
end

function RuleDefiner:haveStartedRule()
	local currentRule = self:getCurrentRule()

	if (currentRule == nil) then
		return false
	else
		return currentRule.selector ~= nil and currentRule.styles == nil
	end
end

function RuleDefiner:haveEndedRule()
	local currentRule = self:getCurrentRule()

	if (currentRule == nil) then
		return true
	else
		return currentRule.selector ~= nil and currentRule.styles ~= nil
	end
end

function RuleDefiner:beginRule(selectorString)
	if self:haveEndedRule() == false then
		error(
			'Rule does not have a body: ' .. self:getCurrentRule().selector:getRawString() .. '. ' ..
			'Please call \'apply\', supplying a modifier function.'
		)
	end

	local selector = self._parser:parse(selectorString)
	local rule = Rule.new(selector, self._currentStylesheetPath)

	table.insert(self._rules, rule)
end

function RuleDefiner:endRule(styles)
	if self:haveStartedRule() == false then
		error(
			'No selector has been specified. ' ..
			'Please call \'to\' before calling \'apply\'.'
		)
	end

	self:getCurrentRule().styles = styles
	self._finalizer:finalize(self:getCurrentRule())
end

function RuleDefiner:mapStylesheetToPartition(stylesheetPath, partitionName)
	local partitions = self._partitionMappings[stylesheetPath]
	
	if partitions == nil then
		partitions = {}
		self._partitionMappings[stylesheetPath] = partitions
	end
	
	table.insert(partitions, partitionName)
end

function RuleDefiner:getPartitionMappingsForStylesheet(stylesheetPath)
	return self._partitionMappings[stylesheetPath]
end

function RuleDefiner:getNumPartitionMappings()
	local count = 0
	for stylesheetPath, partitions in pairs(self._partitionMappings) do
		count = count + #partitions
	end
	return count
end

function RuleDefiner:exportRules()
	return self._exporter:convertToTableRepresentation(
			self._rules,
			self._partitionMappings)
end

return RuleDefiner



