-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local StyleApplicationPhases = plugins.style:lazyRequire('rules.styles.StyleApplicationPhases')
local StyleListDeltaCalculator = plugins.style:lazyRequire('rules.styles.StyleListDeltaCalculator')
local PropertySetter = plugins.style:lazyRequire('rules.styles.definitions.PropertySetter')
local PropertyGetter = plugins.style:lazyRequire('rules.styles.definitions.PropertyGetter')
local PropertyDefaults = plugins.style:lazyRequire('rules.styles.definitions.PropertyDefaults')
local Counters = plugins.style:lazyRequire('profiling.Counters')
local RuleApplicationStages = plugins.style:lazyRequire('rules.RuleApplicationStages')

-- can't lazy load because of equality test below
local PropertyStyle = plugins.style:lazyRequire('rules.styles.definitions.PropertyStyle')

local EffectStyle = plugins.style:lazyRequire('rules.styles.definitions.EffectStyle')
local TableUtils = plugins.style:lazyRequire('utils.TableUtils')
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')

local _firstNodeIndexIsLowerThanSecondNodeIndex

local RuleApplier = class(function(self, context)

	self._enableViewProperties = context:isViewsSupportEnabled()

	self._matchRegistry  = context:getMatchRegistry()
	self._matchingEngine = context:getMatchingEngine()
	self._matchSorter    = context:getMatchSorter()

	self._styleRegistry = context:getStyleRegistry()
	self._styleDeltaCalculator = StyleListDeltaCalculator.new(self._styleRegistry)

	self._bindingManager = context:getBindingManager()
	self._propertySetter = PropertySetter.new(context)

	self._statsManager = context:getStatsManager()
	self._mutationTracker = context:getMutationTracker()

end)

local SORTING_PHASE_ACTION_NAME = 'style.applyRules.lua.sortingPhase'
local MATCHING_PHASE_ACTION_NAME = 'style.applyRules.lua.matchingPhase'
local APPLICATION_PHASE_ACTION_NAME = 'style.applyRules.lua.applicationPhase'

-- Iterates over the supplied rule set and node list to find nodes which match 
-- the rules, and then applies the styles from each matching rule in turn.
function RuleApplier:applyRulesToNodes(ruleSet, nodes, mutations,
		watcherUpdates)
	
	Counters.TOTAL_NODES:setCount(#nodes)
	Counters.TOTAL_RULES:setCount(ruleSet:getNumRules())
		
	-- Ensure that styles are applied to nodes in the correct order - this is
	-- important when styles reference a property of the previous node.
	self._statsManager:startTimelineAction(SORTING_PHASE_ACTION_NAME)
	nodes = self:_sortNodesByChildIndex(nodes)
	self._statsManager:stopTimelineAction(SORTING_PHASE_ACTION_NAME)

	local node
	local matches
	local haveChanged
	local styleDeltas = {}

	-- Matching phase: determines which rules match each node in the list
	self._statsManager:startTimelineAction(MATCHING_PHASE_ACTION_NAME)
	for i = 1, #nodes do
		node = nodes[i]

		-- Get a list of all matches for the current node, sorted in order of
		-- their matching expression's specificity
		matches, haveChanged = self:getSortedMatches(node, ruleSet)
	
		-- Calculate the difference between the styles we've just matched and
		-- any previous styles which have been applied to the node
		if haveChanged then
  			styleDeltas[i] = self._styleDeltaCalculator:calculateDelta(node, matches)
		end
	end
	self._statsManager:stopTimelineAction(MATCHING_PHASE_ACTION_NAME)

	-- Application phase: applies the matches to each node
	self._statsManager:startTimelineAction(APPLICATION_PHASE_ACTION_NAME)
	for i = 1, #nodes do
		node = nodes[i]
		
		-- Add, remove or update styles as necessary based on any change in
		-- matched styles resulting from the matching process. Note that 
		-- in the case where no matches have changed during this tick, no
		-- style delta needs to be calculated and so the entry will be nil.
		if styleDeltas[i] ~= nil then
  			self:_processStyleDelta(node, styleDeltas[i], watcherUpdates)
			self:_flagProgress(node, RuleApplicationStages.STYLE_DELTA_PROCESSING)
			
  		-- In the case where no matches changed, we still need to see whether
  		-- any static bindings need to be relinked - this happens when nodes are
  		-- moved around in the graph such that they still match the same rules
  		-- as before, but their surrounding nodes have changed and so bindings
  		-- that mention `previous.left` or `parent.width` etc need to be updated.
		else
			self._bindingManager:relinkPropertyRuntimeExpressionBindingsForNode(node)
			self:_flagProgress(node, RuleApplicationStages.BINDING_RELINKING)
			
		end
	end
	self._statsManager:stopTimelineAction(APPLICATION_PHASE_ACTION_NAME)

end

function RuleApplier:_flagProgress(node, stage)
	self._mutationTracker:flagProgressForNode(node, stage)
end

-- Given a list of nodes, sorts them in ascending order by their index within
-- the children array of their parent.
function RuleApplier:_sortNodesByChildIndex(nodes)

	-- Copy the nodes list first, so that we dont mutate the input array
	local nodesCopy = {}
	for i, node in ipairs(nodes) do
		nodesCopy[i] = node
	end

	table.sort(nodesCopy, _firstNodeIndexIsLowerThanSecondNodeIndex)
	
	return nodesCopy

end

-- Sorting function used by _sortNodesByChildIndex(). Returns true if the first
-- node has lower child index than the second, which results in the nodes array
-- being sorted in ascending order.
function _firstNodeIndexIsLowerThanSecondNodeIndex(first, second)

	return first:getIndexInParent() < second:getIndexInParent()

end

-- Returns a list of all matches for the current node, sorted in order of their
-- matching expression's specificity
function RuleApplier:getSortedMatches(node, ruleSet)

	-- Get a list of all matches for the current node
	local matches = self._matchingEngine:getMatchingRules(node, ruleSet)

	-- Sort by specificity so that we apply styles in the right order
	self._matchSorter:sortByMatchingExpressionSpecificity(matches)

	-- Update the match registry with the new set of matches
	local haveChanged = self._matchRegistry:setMatchesForNode(node, matches)
	
	if haveChanged then
		Counters.NODES_FOR_WHICH_MATCHES_HAVE_CHANGED:increment()
	else
		Counters.NODES_FOR_WHICH_MATCHES_HAVENT_CHANGED:increment()
	end

	return matches, haveChanged

end

-- Given a delta as generated by StyleListDeltaCalculator, loops over the
-- separate remove, superseded and added groups of styles to add/remove them
-- from the node as necessary.
function RuleApplier:_processStyleDelta(node, styleDelta,
		watcherUpdates)

	local removedStyle
	local supersededStyle
	local activeStyles

	-- Remove any styles which no longer match, and reset the corresponding
	-- properties to their default values
	for j = 1, #styleDelta.removed do
		Counters.STYLES_REMOVED:increment()
		
		removedStyle = styleDelta.removed[j]
		removedStyle:removeFromNode(node)

		self:_restoreDefaultValue(node, removedStyle)
	end

	-- Also remove any styles which have been superseded by other styles and
	-- so are not relevant anymore
	for j = 1, #styleDelta.superseded do
		Counters.STYLES_REMOVED:increment()
		
		supersededStyle = styleDelta.superseded[j]
		supersededStyle:removeFromNode(node)
	end

	-- Apply all active property and effect styles to the node
	--
	-- TODO (pcowburn) it'd be better to just iterate over the list of styles
	-- in delta.added here, in order to prevent re-applying styles that have
	-- already been applied on an earlier tick. However, this currently causes
	-- issues with reflow as it's sometimes dependent on the re-application of
	-- some rules in order to re-evaluate any PropertyRuntimeExpressions which
	-- make up the reflow.
	activeStyles = self._styleRegistry:getActiveStylesForNode(node)
	self:_setProperties(activeStyles.properties, node, styleDelta.added,
			watcherUpdates)
	self:_setEffects(activeStyles.effects, node, styleDelta.added)
	self:_setViewProperties(activeStyles.viewProperties, node, styleDelta.added)
end

-- Called when an effect no longer matches a node, to restore the default value
-- of the corresponding property or effect.
function RuleApplier:_restoreDefaultValue(node, style)

	local styleType = getmetatable(style)

	-- Reset property values
	if styleType == PropertyStyle then

		-- Reset the property back to its default value so that the node regains
		-- its unstyled appearance.
		local propertyName = style:getName()
		local defaultValue = PropertyDefaults[propertyName]

		if defaultValue ~= nil then
			local prevValue = self._styleRegistry:getLastAppliedValue(node, style)

			-- If the previously applied value was an animation, animate out of it.
			if AnimationTrigger.isAnimationTrigger(prevValue) then
				prevValue:cloneWithNewToValue(defaultValue):trigger()
			-- Otherwise, we can just set the property back to its default value.
			else
				self._propertySetter:setPropertyOfNode(node, propertyName, defaultValue)
			end
			
			self:_flagStyleApplication(node, style, defaultValue)
		else
			error('Could not determine default value for ' .. propertyName)
		end

	-- Reset effect values
	elseif styleType == EffectStyle then

		-- Nodes start with no effects by default, so restoring the default
		-- style here means simply removing the effect from the supplied node
		node:removeEffect(style:getHandle())

	end

end

function RuleApplier:_flagStyleApplication(node, style)
	self._mutationTracker:flagStyleApplicationForNode(node, style)
end

-- Applies styles which are specified as properties, i.e. "x = 10". These are set
-- directly on the node, by transforming the property name (i.e. 'x') into a setter
-- name (i.e. 'setX') and calling it.
function RuleApplier:_setProperties(properties, node, addedStyles,
		watcherUpdates)

	local directive

	if self._enableViewProperties then
		directive = context:getNodesRegistry():getViewElement(node:getId())
	end

	self._propertyGetter = self._propertyGetter or PropertyGetter.new()
	local oldValue = nil
	local newValue = nil

	for name, property in pairs(properties) do
		Counters.PROPERTY_STYLES_APPLIED:increment()
		
		if directive and directive:hasWatchers(name) then
			oldValue = self._propertyGetter:getPropertyOfNode(node, name)
		end
		
		property:applyToNode(node, self:_determinePhase(property, addedStyles))
			
		self:_flagStyleApplication(node, property)
			
		-- Notify the node watchers if one of its properties changes
		if directive and directive:hasWatchers(name) then
			newValue = self._propertyGetter:getPropertyOfNode(node, name)

			if oldValue ~= newValue then
				local watchers = directive:getWatchers(name)
				for propertyName, watcher in ipairs(watchers) do
					for _, watchingView in ipairs(watchers) do
						if not watcherUpdates[watchingView] then
							watcherUpdates[watchingView] = {}
						end
						watcherUpdates[watchingView][name] = true
					end
				end
			end
		end
	end

end

-- Applies effects to the node by adding new effects, removing existing ones,
-- or updating properties of effects that are already present.
function RuleApplier:_setEffects(effects, node, addedStyles)

	-- Check that the rule actually specifies some effects (it's not mandatory).
	if effects ~= nil then
		for i, effect in pairs(effects) do
			Counters.EFFECT_STYLES_APPLIED:increment()
			
			effect:applyToNode(
					node, 
					self:_determinePhase(effect, addedStyles), 
					self._mutationTracker)
		end
	end

end

function RuleApplier:_setViewProperties(viewProperties, node, addedStyles)
	if not self._enableViewProperties then
		return
	end

	-- Check that the rule actually specifies some viewProperties (optional).
	if viewProperties ~= nil then
		for i, property in pairs(viewProperties) do
			Counters.VIEW_PROPERTY_STYLES_APPLIED:increment()
			
			property:applyToNode(
					node, self:_determinePhase(property, addedStyles))
					
			self:_flagStyleApplication(node, property)
		end
	end
end

-- Determines whether the style is being applied as a completely new match or
-- as an update via a more specific rule.
function RuleApplier:_determinePhase(style, addedStyles)

	local isNewAddition = TableUtils.arrayContainsValue(addedStyles, style)

	if isNewAddition then
		return StyleApplicationPhases.INITIAL_APPLICATION
	else
		return StyleApplicationPhases.SPECIFICITY_UPDATE
	end

end

return RuleApplier



