-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

--[[

	Interface designed for use in the Lua state that styles are applied within,
	i.e. the state the application maintains at runtime and supplies scene nodes
	to when their matching characteristics change.

  ]]--

local function init()
	-- Set the plugin alias so we may convieniently load scripts usin:
	-- plugins.style:require(...)
	plugins.setPluginAlias('style', 'com.amazon.ignition.framework.style.logic');

	local AnimationLibrary = plugins.style:lazyRequire('expressions.definitions.animations.AnimationLibrary')
	local IteratorsLibrary = plugins.style:lazyRequire('expressions.definitions.iterators.IteratorsLibrary')
	local StyleApplicationContext = plugins.style:lazyRequire('rules.styles.StyleApplicationContext')
	local RuleTableImporter = plugins.style:lazyRequire('rules.RuleTableImporter')
	local MutationConfig = plugins.style:lazyRequire('mutations.MutationConfig')
	local Counters = plugins.style:lazyRequire('profiling.Counters')

	-- Load the animation functions into the global namespace
	AnimationLibrary.loadAnimationFunctions()

	-- Context
	local _context = StyleApplicationContext.new()

	-- Iterators are imported from the Views plugin, so we only load them if
	-- views support is enabled.
	if _context:isViewsSupportEnabled() then
		IteratorsLibrary.loadIteratingFunctions()
	end

	-- Matching
	local _matchRegistry = _context:getMatchRegistry()
	local _matchabilityCache = _context:getMatchabilityCache()

	-- Rules
	local _ruleApplier = _context:getRuleApplier()
	local _ruleSet = _context:getRuleSet()
	local _ruleTableImporter = RuleTableImporter.new(_context)

	-- Expressions
	local _bindingManager = _context:getBindingManager()
	local _bridge = _context:getBridge()
	local _watcherUpdates = {}

	-- Styles
	local _styleRegistry = _context:getStyleRegistry()

	-- Serializers
	local _nodeMatchesSerializer = plugins.style:lazyRequire('serializers.NodeMatchesSerializer')
	local _ruleSerializer = plugins.style:lazyRequire('serializers.RuleSerializer')

	-- Debug
	local _mutationTracker = _context:getMutationTracker()
	local _mutationGraphPrinter = _context:getMutationGraphPrinter()
	local _statsManager = _context:getStatsManager()
	local _shouldPrintStats = false

	addRules = function(rulesContainer)
		-- Reverse the action performed by RuleTableExporter to convert the
		-- rules back into classes now that we've copied them from the rule
		-- definition Lua state.
		_ruleTableImporter:convertFromTableRepresentation(rulesContainer)

		_ruleSet:importRules(rulesContainer)
		return _ruleSet
	end

	clearRules = function()
		_matchabilityCache:clearAll()
		_ruleSet:clear()
	end

	getNumRules = function()
		if _ruleSet ~= nil then
			return _ruleSet:getNumRules()
		else
			return 0
		end
	end

	getNumGlobalRules = function()
		if _ruleSet ~= nil then
			return _ruleSet:getNumGlobalRules()
		else
			return 0
		end
	end

	getNumSelectorSegments = function()
		if _ruleSet ~= nil then
			return _ruleSet:getNumSegments()
		else
			return 0
		end
	end

	addVariables = function(bridgeContainer)
		_bridge:importVariables(bridgeContainer)
		return _bridge
	end

	clearVariables = function()
		_bridge:clear()
	end

	getNumVariables = function()
		if _bridge ~= nill then
			return _bridge:getNumVariables()
		else
			return 0
		end
	end

	getRequiredMutationTypes = function()
		return MutationConfig.REQUIRED_MUTATION_TYPES
	end

	getPropagatedMutationTypes = function()
		return MutationConfig.PROPAGATED_MUTATION_TYPES
	end

	-- Builds a list of matched styles for a given node
	buildRuleMatchesForNode = function(node)
		if (_ruleSet ~= nil) then

			-- Apply all rules to the nodes we've been supplied
			local matches = _ruleApplier:getSortedMatches(node, _ruleSet)

			-- Now serialize the matches we've retrieved
			local jsonPayload = _nodeMatchesSerializer.serialize(node, matches)

			return jsonPayload
		end

		return ""
	end

	-- Builds a Json payload for given ruleIds
	buildRuleDescriptions = function(ruleIds)
		if (_ruleSet ~= nil) then

			local jsonPayload = ""
			for key,id in pairs(ruleIds) do
				jsonPayload = jsonPayload .. _ruleSerializer.serialize(id,
						_ruleSet:getAllRules())

				if (key < #ruleIds) then
					jsonPayload = jsonPayload .. ", "
				end
			end

			return jsonPayload
		end

		return ""
	end

	-- Retrieves the styling properties of a rule by id
	getStyleForRule = function(viewScope, ruleId)
	end

	-- Applies any matching rules to the supplied list of nodes, which will
	-- result in styling updates if matches are found.
	applyRulesToNodes = function(nodes, mutations)
		local counts

		if (_ruleSet ~= nil) then
			_mutationTracker:beginTracking(_ruleSet, nodes, mutations)

			-- Apply all rules to the nodes we've been supplied
			_ruleApplier:applyRulesToNodes(_ruleSet, nodes, mutations,
					_watcherUpdates)

			-- Check whether any mutations that relate to watched properties
			-- have happened, and if so, update the watchers
			_bindingManager:updateStaticWatchersAffectedByMutations(nodes, mutations)

			local mutationTrackingData = _mutationTracker:endTracking()
			_mutationGraphPrinter:printGraph(mutationTrackingData, _context:getTickIndex())

			-- Print stats - we only print if at least one node was supplied,
			-- otherwise it just generates a ton of noise.
			if #nodes > 0 and _statsManager:countersAreEnabled() then
				-- Flag that we need to print the stats when 'updateLiveWatchers'
				-- is called (it must be printed there as the stats may
				-- be affected after the update)
				_shouldPrintStats = true
				counts = Counters.getCounts()
			end
		end

		return counts
	end

	updateLiveWatchers = function()
		-- Check whether any live watchers changed during the last tick
		_bindingManager:updateLiveWatchers(_watcherUpdates)
		notifyViews(_watcherUpdates)
		_watcherUpdates = {}

		if _shouldPrintStats then
			_statsManager:printCounters()
			_statsManager:resetCounters()
			_shouldPrintStats = false
		end
	end

	-- Notify views that one of its nodes changed
	notifyViews = function(watcherUpdates)
		if _context:isViewsSupportEnabled() then
			for view, propertiesChangedSet in pairs(watcherUpdates) do
				local propertiesChanged = {}
				for propertyName, _ in pairs(propertiesChangedSet) do
					table.insert(propertiesChanged, propertyName)
				end
				view:onNodePropertyChange(propertiesChanged)
			end
		end
	end

	-- Removes any references to nodes indicated by the supplied list of node
	-- IDs (called when the corresponding nodes have been deleted).
	cleanUpNodes = function(nodeIds)
		for i = 1, #nodeIds do
			local nodeId = nodeIds[i]

			_bindingManager:removeBindingsByNodeId(nodeId)
			_matchRegistry:removeMatchesForNodeId(nodeId)
			_styleRegistry:removeStylesAndValuesForNodeId(nodeId)
		end
	end

	enableTimelining = function()
		_context:getStatsManager():enableTimelining()
	end

	enableCounters = function()
		_context:getStatsManager():enableCounters()
	end

	enableMutationGraphPrinting = function()
		_context:getMutationTracker():enable()
	end

	setTickIndex = function(tickIndex)
		_context:setTickIndex(tickIndex)
	end

	willDispatchSignal = function(signalName)
		return _context:getSignalQueue():isQueued(signalName)
	end

	retrieveAndClearQueuedSignals = function()
		return _context:getSignalQueue():retrieveAndClearCompletedSignals()
	end

end

init()
