-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
--[[
--
--    Builds the global namespace for the style extension.
--
--    This function also returns a table containing protected management methods
--    for the extension.
--
--]]
return function (namespace)
	assert(type(namespace) == 'string', 'Namespace must be a string')
	assert(_G[namespace] == nil, 'Global style namespace must not already exist')

	local style = {}
	_G[namespace] = style

	local _ruleDefinitionManager
	local _ruleApplicationManager
	local _bridgeManager

	-- Loads the stylesheet specified by the supplied path.
	function style.loadStylesheet(path, partition)
		local res = _ruleDefinitionManager:loadRulesFromFile(path, partition)
		if res ~= true then
			error("Error loading stylesheet "..path)
		end
	end

	-- Queues the loading of the stylesheet specified by the supplied path.
	function style.queueStylesheet(path, partition)
		_ruleDefinitionManager:queueLoadRulesFromFile(path, partition)
	end

	-- Flushes the queue of stylesheets built up by queueStylesheet().
	function style.flushStylesheetQueue()
		local res = _ruleDefinitionManager:flushFileQueue()
		if res ~= true then
			error("Error loading stylesheet")
		end
	end

	-- Forces an immediate application of the style rules to any nodes mutated
	-- during the current tick.
	--
	-- Note that this should only be used when absolutely necessary - styles
	-- will be automatically applied at the end of each tick, and so if possible
	-- applications should wait for this to happen rather than forcing an early
	-- pass via this method.
	function style.applyRules()
		_ruleApplicationManager:applyRules()
	end

	-- Gets the value of the Bridge variable associated with the supplied name.
	function style.getVariable(name)
		local reflectable = _bridgeManager:getVariable(name)
		if reflectable:isEmpty() then
			error("Unsupported variable type when calling getVariable()")
		end
		return reflectable:get()
	end

	-- Sets the value of the Bridge variable associated with the supplied name.
	function style.setVariable(name, value)
		local reflectable = ReflectableValueUserData.new()
		reflectable:set(value)
		local status = _bridgeManager:setVariable(name, reflectable)

		if status == false then
			error("Error setting variable "..name)
		end
	end

	-- Functions for the native extension to call in order to configure various
	-- aspects of the style namespace.
	return
	{
		-- Sets the RuleDefinitionManager instance in use by the Style plugin.
		setRuleDefinitionManager = function (value)
			_ruleDefinitionManager = value
		end,

		-- Sets the RuleApplicationManager instance in use by the Style plugin.
		setRuleApplicationManager = function (value)
			_ruleApplicationManager = value
		end,

		-- Sets the BridgeManager instance in use by the Style plugin.
		setBridgeManager = function (value)
			_bridgeManager = value
		end
	}
end