-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local _iteratorCache = {}

local function _getOrAddToCache(iteratorClassName)
	local iteratorClass = _iteratorCache[iteratorClassName]
	if not iteratorClass then
		iteratorClass = plugins.views:lazyRequire(iteratorClassName)
		_iteratorCache[iteratorClassName] = iteratorClass
	end
	return iteratorClass
end

local function createSpacingIterator(iteratorClassName, ...)
	local iteratorClass = _getOrAddToCache(iteratorClassName)
	local args = {...}
	assert(#args == 1, "Spacing iterators currently take exactly one " ..
			"constructor argument correponding to the 'spacing' " ..
			"between elements. Got ", #args, " instead.")
	return iteratorClass.new({
		spacing = args[1]
	})
end

-- Note: the iterator classes are lazily loaded because this file is used
-- by both RuleApplier and RuleDefiner, but RuleDefiner just needs the keys.
-- We don't want to expose classes from PluginViews to RuleDefiner, as the
-- user would have access to members/classes "private" to the middleware.
return {
	["FixedSpacingHorizontal"] = function(...)
		return createSpacingIterator("properties.iterator.FixedSpacingHorizontal", ...)
	end,
	["FixedSpacingVertical"] = function(...)
		return createSpacingIterator("properties.iterator.FixedSpacingVertical", ...)
	end,
	["CentralDynamicSpacingHorizontal"] = function(...)
		return createSpacingIterator("properties.iterator.CentralDynamicSpacingHorizontal", ...)
	end,
	["CentralDynamicSpacingVertical"] = function(...)
		return createSpacingIterator("properties.iterator.CentralDynamicSpacingVertical", ...)
	end,
	["DynamicSpacingHorizontal"] = function(...)
		return createSpacingIterator("properties.iterator.DynamicSpacingHorizontal", ...)
	end,
	["DynamicSpacingVertical"] = function(...)
		return createSpacingIterator("properties.iterator.DynamicSpacingVertical", ...)
	end,
	["NoIterator"] = function(...)
		local iteratorClass = _getOrAddToCache("properties.iterator.NoIterator")
		return iteratorClass.new()
	end
}
