-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local IExpressionBuilder = plugins.style:lazyRequire("expressions.definitions.builders.IExpressionBuilder")
local AnimationRuntimeExpression = plugins.style:lazyRequire("expressions.definitions.AnimationRuntimeExpression")

local AnimationRuntimeExpressionBuilder = class(IExpressionBuilder,
		function(self, easingFunctionName)
	self._options = {}
	self._options.easingFunctionName = easingFunctionName
end)

-- The AnimationRuntimeExpressionBuilder allows an easing function describing an animation
-- to be defined in a semantic builder-like syntax. For backwards compatibility reasons,
-- there are two possible use cases for specifying animations in this way:
--
-- Eg. 1 - Animate z-value to 1.0 over 0.8s with outQuad easing function after 0.5s delay
--
--     z = OutQuad():to(1.0):duration(0.8):delay(0.5)
--
-- Eg. 2 - Scroll a view's list items with interval between items of 0.5s with easing
--         function of Cubic Bezier with (x1, y1, x2, y2) specified by the given params argument
--
--     viewProperties = {
--         scrollingAnimation = CubicBezier():params(.65, .4, .35, .6):duration(0.5)
--     }

function AnimationRuntimeExpressionBuilder:build()
	if self:_hasValueAnimatingOptions() then
		return AnimationRuntimeExpression.new(
			self._options.from,
			self._options.to,
			self._options.duration,
			self._options.delay,
			self._options.easingFunctionParameters,
			self._options.easingFunctionName,
			self._options.onCompleteSignalName
		)
	else
		return self._options
	end
end

function AnimationRuntimeExpressionBuilder:_hasValueAnimatingOptions()
	return self._options.to or self._options.delay
end

function AnimationRuntimeExpressionBuilder:params(...)
	params = {...}
	self:_assertBuilderOption("easingFunctionParameters", params)
	self._options.easingFunctionParameters = params
	return self
end

function AnimationRuntimeExpressionBuilder:from(value)
	self:_assertBuilderOption("from", value)
	self._options.from = value
	return self
end

function AnimationRuntimeExpressionBuilder:to(value)
	self:_assertBuilderOption("to", value)
	self._options.to = value
	return self
end

function AnimationRuntimeExpressionBuilder:duration(duration)
	self:_assertBuilderOption("duration", duration)
	self._options.duration = duration
	return self
end

function AnimationRuntimeExpressionBuilder:delay(delay)
	self:_assertBuilderOption("delay", delay)
	self._options.delay = delay
	return self
end

function AnimationRuntimeExpressionBuilder:dispatchOnComplete(signalName)
	self:_assertBuilderOption("onCompleteSignalName", signalName)
	self._options.onCompleteSignalName = signalName
	return self
end

local function _assertNotAlreadySet(optionName, optionValue)
	if optionValue ~= nil then
		error("The '" .. optionName .. "' option must not be set more than once")
	end
end

local function _assertNotNil(optionName, argumentValue)
	if argumentValue == nil then
		error("The '" .. optionName .. "' option must not be nil")
	end
end

function AnimationRuntimeExpressionBuilder:_assertBuilderOption(optionName, newValue)
	_assertNotAlreadySet(optionName, self._options[optionName])
	_assertNotNil(optionName, newValue)
end

return AnimationRuntimeExpressionBuilder
