-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local MathematicalRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.MathematicalRuntimeExpression')

local EXPRESSION_TYPE = RuntimeExpressionTypes.ZERO_GUARD

--
-- RuntimeExpression used when trying to guard against dividing by zero, 
-- for example by saying:
--
--    to "ImageNode.square" apply
--    {
--        x = node.x / ZeroGuard(previous.x, 1) <<< If previous.x equals zero
--                                                  then 1 will be returned
--    }
--
local ZeroGuardRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	-- Constructor
	function(self, guarded, default)

		if default == 0 then
			error("The default value for a ZeroGuard cannot be 0")
		end

		self:_init(EXPRESSION_TYPE)
		MathematicalRuntimeExpression.addOperatorOverloads(self)

		self._guarded = guarded
		self._default = default

	end
)

-- Type constant identifying ZeroGuardRuntimeExpressions.
ZeroGuardRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

-- Resolves value of the runtime expression by reading the corresponding
-- property from the node to which it was applied.
function ZeroGuardRuntimeExpression:evaluate(currentProperty, currentNode, phase)

	local value = self._guarded

	if AbstractRuntimeExpression.isExpression(self._guarded) then
		value = self._guarded:evaluate(currentProperty, currentNode, phase)
	end
	
	if value == 0 then
		return self._default
	else
		return value
	end

end

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function ZeroGuardRuntimeExpression:toString()

	local fullName = ZeroGuardRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local defaultDetails   = 'with default value \'' .. self._default .. '\''

	return fullName .. ' ' .. defaultDetails

end

function ZeroGuardRuntimeExpression:getGuarded()
	return self._guarded
end

function ZeroGuardRuntimeExpression:getDefault()
	return self._default
end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function ZeroGuardRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = ZeroGuardRuntimeExpression.EXPRESSION_TYPE,
		guarded = self._guarded,
		default = self._default
	}

end

-- Performs the reverse of toTable(), for use in creating ZeroGuardRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function ZeroGuardRuntimeExpression.fromTable(table)

	return ZeroGuardRuntimeExpression.new(
		table.guarded,
		table.default)

end

return ZeroGuardRuntimeExpression



