-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ffi = lazyRequire('ffi')

local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local MathematicalRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.MathematicalRuntimeExpression')
local PropertyTypes = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')

local _friendlyNamesToFFINames

local EXPRESSION_TYPE = RuntimeExpressionTypes.TYPE

--
-- RuntimeExpression used for specifying that a style value should be a complex
-- type, e.g a Vec2/3/4 or a Color. For example:
--
--    to "ImageNode.square" apply
--    {
--        position = Vec3(1.0, 1.0, 1.0)   <<< Vec3 must be converted to cdata at runtime
--    }
--
local TypeRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	-- Constructor (called when the type is assigned in the definition state).
	--
	-- The friendlyTypeNam argument is the name of the type that the stylesheet
	-- author will be familiar with - 'Vec2', 'Vec3' etc. In some cases this will
	-- be different from the corresponding FFI struct type that gets used when
	-- the style is actually applied - for example, 'Vec2' becomes 'LuaVec2' as
	-- that's the name of the struct exposed by PluginScene's Lua bindings.
	function(self, friendlyTypeName, values)

		self:_init(EXPRESSION_TYPE)
		MathematicalRuntimeExpression.addOperatorOverloads(self)

		self._friendlyTypeName = friendlyTypeName
		self._ffiTypeName = _friendlyNamesToFFINames[friendlyTypeName]
		self._values = values

	end
)

-- Type constant identifying TypeRuntimeExpressions.
TypeRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

-- Lookup table to convert friendly type names to Lua type names.
_friendlyNamesToFFINames =
{
	[PropertyTypes.VEC2] = 'LuaVec2',
	[PropertyTypes.VEC3] = 'LuaVec3',
	[PropertyTypes.VEC4] = 'LuaVec4',
	[PropertyTypes.RECT] = 'LuaRect',
	[PropertyTypes.PIVOT_MODE] = 'LuaPivotMode'
}

-- Resolves the value of the runtime expression by creating a CDATA instance
-- with the values supplied when the expression was created.
function TypeRuntimeExpression:evaluate()

	return ffi.new(self._ffiTypeName, self._values)

end

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function TypeRuntimeExpression:toString()

	local fullName = TypeRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local typeDetails = 'for type \'' .. self._friendlyTypeName .. '\''

	return fullName .. ' ' .. typeDetails

end

function TypeRuntimeExpression:getFriendlyTypeName()
	return self._friendlyTypeName
end

function TypeRuntimeExpression:getFFITypeName()
	return self._ffiTypeName
end

function TypeRuntimeExpression:getValues()
	return self._values
end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function TypeRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = TypeRuntimeExpression.EXPRESSION_TYPE,
		friendlyTypeName = self._friendlyTypeName,
		ffiTypeName = self._ffiTypeName,
		values = self._values
	}

end

-- Performs the reverse of toTable(), for use in creating TypeRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function TypeRuntimeExpression.fromTable(table)

	return TypeRuntimeExpression.new(table.friendlyTypeName, table.values)

end

return TypeRuntimeExpression



