-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local MathematicalRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.MathematicalRuntimeExpression')
local StyleApplicationPhases = plugins.style:lazyRequire('rules.styles.StyleApplicationPhases')
local PropertyDefaults = plugins.style:lazyRequire('rules.styles.definitions.PropertyDefaults')
local PropertyGetter = plugins.style:lazyRequire('rules.styles.definitions.PropertyGetter')

local INITIAL_APPLICATION = StyleApplicationPhases.INITIAL_APPLICATION
local BINDING_UPDATE = StyleApplicationPhases.BINDING_UPDATE

local EXPRESSION_TYPE = RuntimeExpressionTypes.PROPERTY

--
-- RuntimeExpression used when the value of one property of a node is assigned
-- to that of another property, for example by saying:
--
--    to "ImageNode.square" apply
--    {
--        x = 100,
--        y = node.x   <<< y must be evaluated by reading the value of x at runtime
--    }
--
local PropertyRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	--
	-- Constructor. Note that the 'nodeKeyword' argument here is an instance of
	-- the particular NodeKeyword type that generated this RuntimeExpression,
	-- such as ParentNodeKeyword, PreviousNodeKeyword etc. When this expression
	-- is evaluated, the nodeKeyword's traverse() method will be used to do the
	-- job of actually finding the node we need to read the property value from.
	--
	function(self, propertyName, nodeKeyword, isLive)

		self:_init(EXPRESSION_TYPE)
		MathematicalRuntimeExpression.addOperatorOverloads(self)

		self._propertyName = propertyName
		self._nodeKeyword = nodeKeyword
		self._isLive = isLive or false

	end
)

-- Type constant identifying PropertyRuntimeExpressions.
PropertyRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

-- Resolves value of the runtime expression by reading the corresponding
-- property from the node to which it was applied.
function PropertyRuntimeExpression:evaluate(currentProperty, currentNode, phase)

	phase = phase or INITIAL_APPLICATION

	-- Get the source node by asking the node keyword that earlier created this
	-- expression to do a traversal for us. See the implementations in each of
	-- the NodeKeyword definitions (CurrentNodeKeyword, ParentNodeKeyword etc)
	-- to see how this works.
	local sourceNode = self._nodeKeyword:traverse(currentNode)

	self:_createBindingToWatchForValueChanges(
			currentProperty,
			currentNode,
			sourceNode)

	return self:_readCurrentPropertyValue(
		currentNode,
		sourceNode,
		phase)

end

-- Reads the current value of the source property from the source node.
function PropertyRuntimeExpression:_readCurrentPropertyValue(
		currentNode,
		sourceNode,
		phase)

	if (sourceNode == nil) then
		return self:_readFromDefaultValuesList()

	elseif (sourceNode == currentNode) and (phase ~= BINDING_UPDATE) then
		return self:_readFromActiveStylesOrNode(sourceNode)

	else
		return self:_readFromNode(sourceNode)

	end

end

-- Create a binding so that subsequent updates to the property will result in
-- this runtime expression being re-evaluated for the current node.
function PropertyRuntimeExpression:_createBindingToWatchForValueChanges(
		currentProperty,
		currentNode,
		sourceNode)

	self._context:getBindingManager():bindPropertyRuntimeExpression(
		sourceNode,
		self,
		currentNode,
		currentProperty,
		self._isLive)

end

-- If the source node was nil (which happens in cases where the style targets
-- a node's parent but the node doesn't have a parent key, or targets a sibling
-- but the node doesn't have any siblings) then we have to return the value from
-- the set of defaults specified by the PropertyDefaults list.
function PropertyRuntimeExpression:_readFromDefaultValuesList()

	local defaultValue = PropertyDefaults[self._propertyName]

	if defaultValue == nil then
		error(
			'Source node was nil when evaluating ' .. self:toString() .. ', ' ..
			'and no default property value exists for \'' .. self._propertyName .. '\''
		)
	else
		return defaultValue
	end

end

-- If the current set of styles being applied contains a value for the property,
-- and the source node is the current node, then we need to read the property value
-- directly from the styles rather than attempting to retrieve it from the node.
--
-- This avoids race conditions whereby an expression references a node property
-- which has not yet had its value applied from the current set of styles.
function PropertyRuntimeExpression:_readFromActiveStylesOrNode(node)

	local activeStyles = self._context:getStyleRegistry():getActiveStylesForNode(node)
	local activeProperty = (activeStyles ~= nil) and (activeStyles.properties[self._propertyName]) or nil

	if activeProperty ~= nil then
		return activeProperty:getValue()
	else
		return self:_readFromNode(node)
	end

end

-- PropertyGetter instance used by the evaluate() method below.
local _propertyGetter = PropertyGetter.new()

-- Reads the property value from the node by using PropertyGetter to translate
-- 'x' into 'getX' and thus invoke the corresponding getter function.
function PropertyRuntimeExpression:_readFromNode(node)

	return _propertyGetter:getPropertyOfNode(node, self._propertyName)

end

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function PropertyRuntimeExpression:toString()

	local fullName = PropertyRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local propertyDetails   = 'for property \'' .. self._propertyName .. '\''
	local sourceNodeDetails = 'and source node type \'' .. self._nodeKeyword:getSourceNodeType() .. '\''

	return fullName .. ' ' .. propertyDetails .. ' ' .. sourceNodeDetails

end

-- Returns the name of the property this expression will read from the source node.
function PropertyRuntimeExpression:getPropertyName()

	return self._propertyName

end

-- Returns the NodeKeyword used by this expression (current, parent etc).
function PropertyRuntimeExpression:getNodeKeyword()

	return self._nodeKeyword

end

-- Returns whether this expression will watch for changes of the target property.
function PropertyRuntimeExpression:isLive()

	return self._isLive

end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function PropertyRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = PropertyRuntimeExpression.EXPRESSION_TYPE,
		propertyName = self._propertyName,
		nodeKeyword = self._nodeKeyword:toTable(),
		isLive = self._isLive
	}

end

-- Performs the reverse of toTable(), for use in creating PropertyRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function PropertyRuntimeExpression.fromTable(table)

	return PropertyRuntimeExpression.new(
		table.propertyName,
		table.nodeKeyword,
		table.isLive)

end

return PropertyRuntimeExpression



