-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local MathematicalRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.MathematicalRuntimeExpression')

local EXPRESSION_TYPE = RuntimeExpressionTypes.BRIDGE

--
-- RuntimeExpression used for specifying that a style should be assigned to
-- the value of a variable belonging to the Bridge. For example:
--
--    to "ImageNode.logo" apply
--    {
--        x = getVariable('logoPositionX')
--    }
--
local BridgeRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	-- Constructor (called when the type is assigned in the definition state).
	function(self, variableName)

		self:_init(EXPRESSION_TYPE)
		MathematicalRuntimeExpression.addOperatorOverloads(self)
		self._variableName = variableName

	end
)

-- Type constant identifying BridgeRuntimeExpressions.
BridgeRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

-- Resolves the value of the runtime expression by retrieving the corresponding
-- variable value from the Bridge.
function BridgeRuntimeExpression:evaluate(currentProperty, currentNode, phase)

	self:_createBindingToWatchForValueChanges(currentProperty, currentNode)

	return self._context:getBridge():getVariable(self._variableName)

end

-- Create a binding so that subsequent updates to the variable will result in
-- this runtime expression being re-evaluated for the current node.
function BridgeRuntimeExpression:_createBindingToWatchForValueChanges(
		currentProperty,
		currentNode)

	self._context:getBindingManager():bindBridgeRuntimeExpression(
		self._context:getBridge(),
		self,
		currentNode,
		currentProperty)

end

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function BridgeRuntimeExpression:toString()

	local fullName = BridgeRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local variableDetails = 'for variable name \'' .. self._variableName .. '\''

	return fullName .. ' ' .. variableDetails

end

function BridgeRuntimeExpression:getVariableName()
	return self._variableName
end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function BridgeRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = BridgeRuntimeExpression.EXPRESSION_TYPE,
		variableName = self._variableName
	}

end

-- Performs the reverse of toTable(), for use in creating BridgeRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function BridgeRuntimeExpression.fromTable(table)

	return BridgeRuntimeExpression.new(table.variableName, table.values)

end

return BridgeRuntimeExpression
