-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local PropertyWatcher = plugins.style:lazyRequire('expressions.bindings.watchers.PropertyWatcher')
local Binding = plugins.style:lazyRequire('expressions.bindings.Binding')

--
-- Stores bindings created by BindingManager.
--
local BindingMap = class(function(self)

	self._staticWatchersById = {}
	self._liveWatchersById = {}
	self._bindingsById = {}
	self._nodesToBindings = {}

end)

-- Returns a list of all bindings as an associative array keyed against the
-- binding ID.
function BindingMap:getBindings()

	return self._bindingsById

end

-- Returns a list of all live watchers as an associative array keyed against
-- the watcher ID.
function BindingMap:getLiveWatchers()

	return self._liveWatchersById

end

-- Returns a list of all static watchers as an associative array keyed against
-- the watcher ID.
function BindingMap:getStaticWatchers()

	return self._staticWatchersById

end

-- Adds the supplied binding to the internal data structure.
function BindingMap:addBinding(binding)

	self:_registerBinding(binding)
	self:_mapNodeToBinding(binding:getNode(), binding)

	local watcher = binding:getWatcher()
	self:_registerWatcher(watcher)

	-- If the watcher is associated with a node (i.e. as is the case with
	-- PropertyWatchers, which watch properties of a particular node), then
	-- we need to create an association between the watched node and the
	-- binding so that when the node eventually dies the binding can be
	-- cleaned up.
	if (watcher.getNode ~= nil) and (watcher:getNode() ~= nil) then
		self:_mapNodeToBinding(watcher:getNode(), binding)
	end

end

-- Returns any bindings associated with the supplied Node.
function BindingMap:getBindingsByNodeId(nodeId)
	return self._nodesToBindings[nodeId]
end

-- Removes any bindings associated with the supplied Node.
function BindingMap:removeBindingsByNodeId(nodeId)

	local bindings = self._nodesToBindings[nodeId]

	self:_destroyBindings(bindings)
	self._nodesToBindings[nodeId] = nil

end

-- Removes any bindings associated with the supplied Node and PropertyStyle.
function BindingMap:removeBindingsByNodeIdAndPropertyStyle(nodeId, propertyStyle)

	local bindings = self._nodesToBindings[nodeId]

	self:_destroyBindings(bindings, propertyStyle)

end

local _bindingCounter = 0

-- Adds the supplied Binding to the _bindingsById table.
function BindingMap:_registerBinding(binding)
	_bindingCounter = _bindingCounter + 1
	binding.id = _bindingCounter
	self._bindingsById[binding.id] = binding
end

local _watcherCounter = 0

-- Adds the supplied Watcher to the _watchersById table.
function BindingMap:_registerWatcher(watcher)
	_watcherCounter = _watcherCounter + 1
	watcher.id = _watcherCounter

	if watcher:isLive() then
		self._liveWatchersById[watcher.id] = watcher
	else
		self._staticWatchersById[watcher.id] = watcher
	end
end

-- Provides a way to clean up bindings when nodes die, by maintaining a map of
-- nodeIds to bindings that reference them.
function BindingMap:_mapNodeToBinding(node, binding)
	local nodeId = node:getId()
	local bindings = self._nodesToBindings[nodeId]

	if bindings == nil then
		bindings = {}
		self._nodesToBindings[nodeId] = bindings
	end

	bindings[binding.id] = binding
end

-- Called when either a node is destroyed or a PropertyStyle which had created
-- a binding no longer matches a given node.
--
-- Optionally a property style can be supplied as the second argument - if this
-- is supplied, then only bindings which match the supplied property style will
-- be destroyed.
function BindingMap:_destroyBindings(bindings, forPropertyStyle)

	if bindings ~= nil then
		for bindingId, binding in pairs(bindings) do
			if (forPropertyStyle == nil)
			or (binding:getPropertyStyle() == forPropertyStyle) then
				self._staticWatchersById[binding:getWatcher().id] = nil
				self._liveWatchersById[binding:getWatcher().id] = nil
				self._bindingsById[binding.id] = nil
				bindings[bindingId] = nil
			end
		end
	end

end

return BindingMap
