-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local PropertyWatcher = plugins.style:lazyRequire('expressions.bindings.watchers.PropertyWatcher')
local BridgeWatcher = plugins.style:lazyRequire('expressions.bindings.watchers.BridgeWatcher')
local Binding = plugins.style:lazyRequire('expressions.bindings.Binding')
local BindingMap = plugins.style:lazyRequire('expressions.bindings.BindingMap')
local MutationTypesToAffectedProperties = plugins.style:lazyRequire('mutations.MutationTypesToAffectedProperties')
local Counters = plugins.style:lazyRequire('profiling.Counters')

--
-- Provides a means to create new watchers and locate those already in existence.
--
-- For unit testing, the binding map used to store bindings can optionally be
-- passed in as the second argument. If this is not supplied however, one will
-- be automatically created.
--
local BindingManager = class(function(self, context, mutationTracker, bindingMap)

	self._context = context
	self._notifyViewsOfWatcherUpdates = context:isViewsSupportEnabled()
	self._statsManager = context:getStatsManager()
	self._mutationTracker = mutationTracker
	self._map = bindingMap or BindingMap.new()

end)

local UPDATE_LIVE_WATCHERS_ACTION_NAME = 'style.applyRules.lua.updateLiveWatchers'
local UPDATE_MUTATED_STATIC_WATCHERS_ACTION_NAME = 'style.applyRules.lua.updateMutatedStaticLiveWatchers'

-- Updates all static watchers for the supplied node ID and property name pair,
-- so that any bindings associated with them can be re-applied. This is done
-- whenever a property value is updated during rule application.
function BindingManager:updateStaticWatchersByNodeIdAndPropertyName(
		nodeId,
		propertyName)

	self:_forEachBinding(nodeId, function(binding, watcher)
		Counters.STATIC_WATCHERS_ASSESSED:increment()
			
		if  (watcher:getNodeId() == nodeId)
		and (watcher:getPropertyName() == propertyName)
		and (watcher:hasChangedSinceLastCall())
		then
			self:_updateBinding(binding, Counters.STATIC_WATCHERS_UPDATED)
		end
	end)

end

function BindingManager:_updateBinding(binding, counter)
	self._mutationTracker:recordBindingUpdate(
			binding:getWatcher():getNode(),
			binding:getWatcher():getPropertyName(),
			binding:getNode(),
			binding:getPropertyStyle():getName(),
			binding:getWatcher():getCachedValue())

	counter:increment()
	binding:update()
end

local ForEachResult = 
{
	COMPLETE = 0,
	BAIL = 1
}

function BindingManager:_forEachBinding(nodeId, callback)
	local bindings = self._map:getBindingsByNodeId(nodeId)

	if bindings ~= nil then
		for i, binding in pairs(bindings) do
			if callback(binding, binding:getWatcher()) == ForEachResult.BAIL then
				return ForEachResult.BAIL
			end
		end
	end

	return ForEachResult.COMPLETE
end

-- Called as part of the rule application process, when the types of mutations
-- that have happened to each node in the last tick is known. Automatically
-- updates any static watchers which watch properties that are likely to have
-- been affected by the mutations that took place - for example, whenever a
-- MutationType.INFERRED_DIMENSIONS_CHANGED mutation happens, we know that the
-- width and height of the image have changed and so any bindings that are 
-- watching these dimensions will need to be updated.
function BindingManager:updateStaticWatchersAffectedByMutations(
		nodes,
		mutations)

	self._statsManager:startTimelineAction(UPDATE_MUTATED_STATIC_WATCHERS_ACTION_NAME)
		
	self:_forEachMutation(nodes, mutations, function(node, mutationType)
		local affectedProperties = MutationTypesToAffectedProperties[mutationType]
		if affectedProperties ~= nil then
			local nodeId = node:getId()
			
			self:_forEachBinding(nodeId, function(binding, watcher)
				Counters.MUTATED_WATCHERS_ASSESSED:increment()
			
				if  (watcher:getNodeId() == nodeId)
				and (affectedProperties[watcher:getPropertyName()] == true)
				and (watcher:hasChangedSinceLastCall())
				then
					self:_updateBinding(binding, Counters.MUTATED_WATCHERS_UPDATED)
				end
			end)
		end
	end)
	
	self._statsManager:startTimelineAction(UPDATE_MUTATED_STATIC_WATCHERS_ACTION_NAME)

end

function BindingManager:_forEachMutation(nodes, mutations, callback)
	for i, node in ipairs(nodes) do
		for j, mutationInfo in ipairs(mutations[i]) do
			local mutationType
		
			-- If the mutation is one that includes mutation metadata (i.e. tag
			-- addition or removal) then it's represented as table. Otherwise
			-- it's just a plain number which corresponds to the mutationType.
			if type(mutationInfo) == 'table' then
				mutationType = mutationInfo.mutationType
			else
				mutationType = mutationInfo
			end

			callback(node, mutationType)
		end
	end
end

-- Asks each live watcher (i.e. those which are checking for external changes
-- to the values they are watching) to check whether the value they are watching
-- has been updated, and if so triggers any associated bindings to be re-applied.
function BindingManager:updateLiveWatchers(watcherUpdates)

	self._statsManager:startTimelineAction(UPDATE_LIVE_WATCHERS_ACTION_NAME)

	local liveWatchers = self._map:getLiveWatchers()

	for id, watcher in pairs(liveWatchers) do
		if watcher:hasChangedSinceLastCall() then
			local name = watcher.binding:getPropertyStyle():getName()

			watcher.binding:update()

			local node = watcher:getNode()
			if node and self._notifyViewsOfWatcherUpdates then
				local directive = context:getNodesRegistry():getViewElement(node:getId())

				if directive and directive:hasWatchers(name) then
					local watchers = directive:getWatchers(name)
					for _, watchingView in ipairs(watchers) do
						if not watcherUpdates[watchingView] then
							watcherUpdates[watchingView] = {}
						end
						watcherUpdates[watchingView][name] = true
					end
				end
			end
		end
	end

	self._statsManager:stopTimelineAction(UPDATE_LIVE_WATCHERS_ACTION_NAME)

end

-- Adds a binding between the supplied PropertyRuntimeExpression and a given
-- node, so that when the property which the PropertyRuntimeExpression relates
-- to is modified, the property style which depends on is re-applied.
function BindingManager:bindPropertyRuntimeExpression(
		sourceNode,
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle,
		isLive)

	if not self:_isDuplicatePropertyBinding(
		sourceNode,
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle,
		isLive)
	then

		local propertyWatcher = PropertyWatcher.new(sourceNode, sourceRuntimeExpression)
		propertyWatcher:setIsLive(isLive)
		
		return self:_createBinding(targetNode, targetPropertyStyle, propertyWatcher)

	end

end

-- Returns true if a binding already exists that matches the supplied values.
--
-- Note that this could be done more elegantly if we just instantiated a new
-- watcher and binding and then tested them for equality against the existing
-- bindings (using an __eq metamethod for the check). However, that would mean
-- creating a lot of garbage as we'd be generating a new bindings and a new
-- watcher and then in the majority of cases throwing them away once we realise
-- they're duplicates, so it makes more sense to do the check here.
function BindingManager:_isDuplicatePropertyBinding(
		sourceNode,
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle,
		isLive)

	local sourcePropertyName = sourceRuntimeExpression:getPropertyName()
	local sourceNodeId = sourceNode:getId()	
	
	local result = self:_forEachBinding(targetNode:getId(), function(binding, watcher)
		if  (watcher:getNodeId() == sourceNodeId)
		and (watcher:getPropertyName() == sourcePropertyName)
		and (watcher:isLive() == isLive)
		and (binding:getPropertyStyle() == targetPropertyStyle)
		then
			return ForEachResult.BAIL
		end
	end)

	return result == ForEachResult.BAIL

end

-- Called by RuleApplier when it detects that the matches for a given node
-- haven't changed, but that its position in the graph might have. Checks
-- whether any of the existing bindings need to be updated based on the
-- new neighbouring nodes, and relinks the watchers if so.
function BindingManager:relinkPropertyRuntimeExpressionBindingsForNode(node)

	self:_forEachBinding(node:getId(), function(binding, watcher)
		if getmetatable(watcher) == PropertyWatcher then
			-- Ignore watchers where the node is observing itself, as these
			-- are not candidates for being relinked.
			if watcher:getNode() ~= node then
				Counters.BINDINGS_ASSESSED_FOR_RELINKING:increment()
	
				self:_retargetWatchedNodeByReperformingTraversal(node, watcher)
			end
		end
	end)

end

function BindingManager:_retargetWatchedNodeByReperformingTraversal(node, watcher)

	-- Retrieve the type of node keyword that was used when the binding was 
	-- created, i.e. `previous` or `parent`.
	local nodeKeyword = watcher:getPropertyRuntimeExpression():getNodeKeyword()
	
	-- Use the node keyword's traverse() method to perform a new traversal 
	-- from the current node to the source node.
	local newSourceNode = nodeKeyword:traverse(node)
	local oldSourceNode = watcher:getNode()
	
	if oldSourceNode:getId() ~= newSourceNode:getId() then
		Counters.BINDING_RELINKINGS_PERFOMED:increment()
				
		watcher:setNode(newSourceNode)
		
		if watcher:hasChangedSinceLastCall() then
			self:_updateBinding(watcher.binding, Counters.BINDINGS_UPDATED_POST_RELINKING)
		end
	end

end

-- Adds a binding between the supplied PropertyRuntimeExpression and a given
-- node, so that when the property which the PropertyRuntimeExpression relates
-- to is modified, the property style which depends on is re-applied.
function BindingManager:bindBridgeRuntimeExpression(
		bridge,
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle)

	if not self:_isDuplicateBridgeBinding(
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle)
	then

		local watchedVariableName = sourceRuntimeExpression:getVariableName()
		local variableWatcher = BridgeWatcher.new(bridge, watchedVariableName)

		return self:_createBinding(targetNode, targetPropertyStyle, variableWatcher)

	end

end

-- Returns true if a binding already exists that matches the supplied values.
function BindingManager:_isDuplicateBridgeBinding(
		sourceRuntimeExpression,
		targetNode,
		targetPropertyStyle)

	local sourceVariableName = sourceRuntimeExpression:getVariableName()
	
	local result = self:_forEachBinding(targetNode:getId(), function(binding, watcher)
		if  (watcher:getVariableName() == sourceVariableName)
		and (binding:getPropertyStyle() == targetPropertyStyle)
		then
			return ForEachResult.BAIL
		end
	end)

	return result == ForEachResult.BAIL

end

-- Creates a binding for the supplied node, propertyStyle and watcher, and adds
-- it to the BindingMap.
function BindingManager:_createBinding(node, propertyStyle, watcher)
	local binding = Binding.new(node, propertyStyle, watcher)

	-- Provide a way to reach the binding from the watcher - this is required
	-- so that the updateLiveWatchers() method can retrieve it.
	watcher.binding = binding

	self._map:addBinding(binding)

	return binding
end

-- Removes any bindings associated with the supplied node.
function BindingManager:removeBindingsByNodeId(nodeId)
	self._map:removeBindingsByNodeId(nodeId)
end

-- Removes any bindings associated with the supplied Node and PropertyStyle.
function BindingManager:removeBindingsByNodeIdAndPropertyStyle(nodeId, propertyStyle)
	self._map:removeBindingsByNodeIdAndPropertyStyle(nodeId, propertyStyle)
end

return BindingManager
