
local PropertyTypes = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')
local PropertyTypeChecker = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypeChecker')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local AnimationRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AnimationRuntimeExpression')
local BridgeRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.BridgeRuntimeExpression')
local su = plugins.style:lazyRequire('utils.StringUtils')

local _bridge
local _checkPropertyDefinitions
local _propertyTypeExists
local _retrievePropertyValuesFromVarArgs
local _checkPropertyValues
local _assertCorrectNumberOfValues
local _insertDefaultPropertyValues
local _assertCorrectValueTypes
local _createPropertiesList
local _checkForTypeMismatch
local _isExpressionOfType
local _unwrapValue

--
-- Factory method for creation of effect definitions, ensuring that each has a NAME
-- constant made available statically and contains a list of effect properties.
--
-- Usage is as follows (example definition shown for the 'Alpha' effect):
--
--    Alpha = EffectFactory.createEffectDefinition('alpha',
--    {
--        {
--            name = "alphaValue",
--            type = PropertyTypes.FLOAT
--        }
--    })
--
-- This will create an effect function which users can use in their stylesheets
-- when specifying that a given node should have the 'alpha' effect applied by the
-- renderer. The effect takes one argument, which must be a floating point number.
--
-- This can then be used as follows:
--
--    to "ImageNode.faded" apply
--    {
--        effects =
--        {
--            fade = Alpha(0.5)
--        }
--    }
--
-- When the stylesheet is interpreted, the Alpha() function will check each of the
-- supplied args and return an effect instance containing the values. This is then
-- used at style application time in order to finally apply the effect to the node.
--
-- Note that the 'effectType' argument here must correspond to the name of the effect as
-- specified in the Renderer plugin. For example, the effectType used for the Alpha()
-- effect is 'alpha' (lowercase) as this corresponds to the assets/effects/alpha.ifx
-- file in the Renderer's assets folder.
--
local function createEffectDefinition(effectType, propertyDefinitions)

	-- The effectType argument is required.
	if (effectType == nil) or (type(effectType) ~= 'string') then
		error("The 'effectType' argument to createEffectDefinition() is required.")
	end

	-- The propertyDefinitions argument is optional; if the effect being defined
	-- doesn't require any arguments then it can be omitted.
	propertyDefinitions = propertyDefinitions or {}

	-- Check that each of the property definitions we've been supplied is valid.
	_checkPropertyDefinitions(propertyDefinitions)

	local Effect = {}
	local metatable = {}

	-- Adds constructor-like behaviour to the effect. This is the function that
	-- stylesheet authors use in their stylesheets when using an effect - e.g.
	--
	--    effects =
	--    {
	--        fade = Alpha(0.5)
	--    }
	--
	-- Here, the 'Alpha()' function is the function created below when the
	-- createEffectDefinition() method is called with 'alpha' as the effectType.
	-- This function iterates over the varargs supplied to it, checking that each
	-- arg is present and matches the expected type as specified in the supplied
	-- propertyDefinitions table.
	metatable.__call = function(self, ...)

		local propertyValues = _retrievePropertyValuesFromVarArgs(...)

		_checkPropertyValues(propertyValues, propertyDefinitions, effectType)

		local properties = _createPropertiesList(propertyValues, propertyDefinitions)

		return
		{
			type = effectType,
			properties = properties
		}

	end

	-- Add the TYPE constant
	Effect.TYPE = effectType

	setmetatable(Effect, metatable)

	return Effect
end

-- Checks that each of the property definitions supplied when defining an effect is
-- valid, i.e. that it has both a 'name' and a 'type' attribute.
function _checkPropertyDefinitions(propertyDefinitions)

	for i, definition in ipairs(propertyDefinitions) do

		if (definition.name == nil) or (type(definition.name) ~= 'string') then
			error('The name attribute of property ' .. i .. ' is missing or invalid.')

		elseif (definition.type == nil)then
			error('The ' .. definition.name .. ' property does not specify a type.')

		elseif (definition.default == nil)then
			error('The ' .. definition.name .. ' property does not specify a default value.')

		elseif (PropertyTypes.isValid(definition.type) == false) then
			error(definition.type .. ' is not a recognised effect property type.')
		end

	end

end

-- Converts the supplied varargs list into an indexed table.
function _retrievePropertyValuesFromVarArgs(...)

	local propertyValues = {}

	for i = 1, select('#', ...) do
		propertyValues[i] = select(i, ...)
	end

	return propertyValues

end

-- Checks that the supplied values matched the expected values as specified when the
-- effect was defined. Checks the number of values supplied and the type of each one.
function _checkPropertyValues(propertyValues, propertyDefinitions, effectType)

	_assertCorrectNumberOfValues(propertyValues, propertyDefinitions, effectType)
	_insertDefaultPropertyValues(propertyValues, propertyDefinitions)
	_assertCorrectValueTypes(propertyValues, propertyDefinitions, effectType)

end

-- Checks the number of property values supplied is not larger than the number of property
-- definitions that were specified when the effect was defined.
function _assertCorrectNumberOfValues(propertyValues, propertyDefinitions, effectType)

	local numValues = #propertyValues
	local numDefinitions = #propertyDefinitions

	-- Check we got the correct number values
	if numValues > numDefinitions then
		error(su.concat(
			su.uppercaseFirst(effectType),
			' expects ',
			numDefinitions,
			' value(s) but ',
			numValues, ' ', su.wasOrWere(numValues),
			' supplied.'
		))

	end
end

-- Inserts the default values for the final unassigned propertyValues
function _insertDefaultPropertyValues(propertyValues, propertyDefinitions)

	for i = #propertyValues+1, #propertyDefinitions do
		propertyValues[i] = propertyDefinitions[i].default
	end

end

-- Checks each property value to ensure if matches the type that is specified in
-- the corresponding property definition.
function _assertCorrectValueTypes(propertyValues, propertyDefinitions, effectType)

	for i, value in ipairs(propertyValues) do

		local propertyDefinition = propertyDefinitions[i]
		local propertyName = propertyDefinition.name
		local expectedType = propertyDefinition.type
		local mismatchType = _checkForTypeMismatch(expectedType, value)

		if mismatchType ~= nil then
			error(su.concat(
				 'The ', su.indexToOrdinal(i), ' value (', propertyName, ')',
				' supplied to ', su.uppercaseFirst(effectType), '()',
				' must be ', su.aOrAn(expectedType), ' ', expectedType,
				' but ', su.aOrAn(mismatchType), ' ', mismatchType,
				' was supplied.'
			))
		end

	end

end

-- Uses PropertyTypeChecker to check whether the supplied value is mismatched
-- against the supplied type. If not (i.e. the types match), returns nil. If
-- there is a mismatch, returns the name of the mismatched type.
function _checkForTypeMismatch(type, value)

	value = _unwrapValue(value)

	-- If the supplied value is a reference to a Bridge variable, we need to
	-- check whether the type of the Bridge variable matches the type the effect
	-- is expecting by looking it up in the Bridge.
	if _isExpressionOfType(value, BridgeRuntimeExpression.EXPRESSION_TYPE) then
		local bridgeVariableType = _bridge:getVariableType(value:getVariableName())

		if bridgeVariableType ~= type then
			return bridgeVariableType
		else
			return nil
		end
	-- For all other value types, we can just ask PropertyTypeChecker to do the
	-- work for us.
	else
		return PropertyTypeChecker.checkForTypeMismatch(type, value)
	end
end

-- Used in cases where the value supplied to an effect property is wrapped in
-- an AnimationRuntimeExpression. Returns the underlying value (i.e. the 'to'
-- value of the animation) so that it can be type checked.
function _unwrapValue(value)
	if _isExpressionOfType(value, AnimationRuntimeExpression.EXPRESSION_TYPE) then
		return value:getTo()
	else
		return value
	end
end

-- Returns true if the supplied value is a runtime expression and matches the
-- supplied type.
function _isExpressionOfType(value, expectedType)
	if  (AbstractRuntimeExpression.isExpression(value))
	and (value:getExpressionType() == expectedType)
	then
		return true
	else
		return false
	end
end

--
-- Creates the list of properties that will be returned when the effect function
-- is called, which marries the values supplied to the effect function back up
-- to the property name and property type that were specified when the effect
-- was originally defined. For example...
--
-- Given propertyValues:
--
--     {
--         42
--     }
--
-- And propertyDefinitions:
--
--     {
--         {
--             name = "myInt",
--             type = PropertyTypes.INT
--         }
--     }
--
-- Returns:
--
--    {
--        myInt =
--        {
--            type  = PropertyTypes.INT,
--            value = 42
--        }
--    }
--
function _createPropertiesList(propertyValues, propertyDefinitions)

	local properties = {}

	for i, value in ipairs(propertyValues) do

		local propertyDefinition = propertyDefinitions[i]

		properties[propertyDefinition.name] =
		{
			type = propertyDefinition.type,
			default = propertyDefinition.default,
			value = value
		}

	end

	return properties

end

-- Sets the Bridge instance in use by the current context. Used by _unwrapValue
-- in order to retrieve the underlying value of a Bridge variable when performing
-- type checking.
local function setBridge(bridge)
	_bridge = bridge
end

-- Public exports.
return
{
	createEffectDefinition = createEffectDefinition,
	setBridge = setBridge
}