
local Asset = plugins.style:lazyRequire('assets.Asset')
local NamespacedAssetGetter = plugins.style:lazyRequire('assets.NamespacedAssetGetter')

local AssetLibrary = class(function(self)
	self._assetsByName = {}

	self._assetPaths = {}
	self._assetGetters = {}
end)

-- Creates a new asset and adds it to the library. If the asset is configured
-- to preload (the default behaviour) then this method will also trigger the
-- preloading process.
function AssetLibrary:defineAsset(attributes)

	local asset = Asset.new(attributes)
	local name = asset:getName()

	if self._assetsByName[name] ~= nil then
		error('An asset with the name ' .. name .. ' has already been defined.')
	end

	self._assetsByName[name] = asset

	self:_preloadIfRequested(asset)

	return asset

end

function AssetLibrary:_preloadIfRequested(asset)

	if asset:getPreload() and self._preloader ~= nil then
		self._preloader:preload(asset:getSource())
	end

end

-- Returns a reference to the asset with the supplied name, or throws an error
-- if no such asset exists.
function AssetLibrary:getAsset(name)

	local asset = self._assetsByName[name]

	if asset == nil then
		error('No asset with the name ' .. name .. ' has been defined.')
	end

	return asset

end

-- Clears all assets in the library
function AssetLibrary:clear()
	self._assetsByName = {}
	self._preloader:clear()
end

-- Sets the preloader to be used for loading assets ahead of time.
function AssetLibrary:setAssetPreloader(preloader)
	self._preloader = preloader
end

function AssetLibrary:registerAssetPath(namespace, path)
	self._assetPaths[namespace] = path
	return self._assetPaths
end

function AssetLibrary:removeAssetPath(namespace)
	self._assetPaths[namespace] = nil
	return self._assetPaths
end

function AssetLibrary:getAssetPath(namespace)
	return self._assetPaths[namespace]
end

function AssetLibrary:getAssetGetterForNamespace(namespace)
	local assetPath = self:getAssetPath(namespace)
	if not assetPath then
		error("No path was registered for namespace '" .. namespace .. "'.")
	end

	local assetGetterInstance = self._assetGetters[namespace]

	if not assetGetterInstance then
		assetGetterInstance = NamespacedAssetGetter.new(self, namespace)
		self._assetGetters[namespace] = assetGetterInstance
	end

	return assetGetterInstance
end

return AssetLibrary



