local ffi = require("ffi")
local interactivityMode = require("external.InteractivityMode")

_G["SceneNodeWrapper"] =
{
	ENABLE_SCENE_GRAPH_CHANGES_FROM_LUA = true
}

_G["scene"] = _G["scene"] or {}
_G["scene"].InteractivityFlag = interactivityMode.flagsEnum
_G["scene"].InteractivityMode = interactivityMode.constructor

local _wrapSceneGraphModifierMethods = function(metatable)
	local sceneGraphModifiers =
	{
		"addChild", "addChildBefore", "addChildAt", "removeChild",
		"addChildAfter", "setParent", "addMaskNode"
	}

	for i, methodName in ipairs(sceneGraphModifiers) do
		local originalMethod = metatable[methodName]
		metatable[methodName] = function(userdata, ...)
			if SceneNodeWrapper.ENABLE_SCENE_GRAPH_CHANGES_FROM_LUA then
				return originalMethod(userdata, ...)
			else
				log.error("SceneNode." .. methodName .. " has been disabled! " .. debug.traceback())
			end
		end
	end
end

local _wrap = function()
	local metatable = getMetatable("SceneNodeUserData")

	if metatable == nil then
		error("Could not get metatable for SceneNode")
	end

	local getChildrenFn = metatable["getChildren"]
	if getChildrenFn == nil then
		error("Could not wrap 'SceneNode::getChildren()', function not found!")
	end

	metatable["getChildren"] = function(userdata, childrenVector)
		local numChildren = userdata:getNumberOfChildren()
		local children = ffi.new('SceneNodeUserData*[' .. numChildren .. ']', {})
		getChildrenFn(userdata, children)

		-- A userdata that is wrapped with ffi.gc *cannot* be put back into the
		-- ffi-allocated pointer vector. The reference counting mechanism loses
		-- track of the object and it can be collected at any time even if there
		-- is a reference to it. It can, though, be put in a table.
		local childrenTable = {}
		for i = 0, (numChildren - 1) do
			childrenTable[i+1] = ffi.gc(children[i], metatable.__gc)
		end
		return childrenTable
	end

	_wrapSceneGraphModifierMethods(metatable)
end

_wrap()
