
if ( typeof(hawaii) == 'undefined' )
{
	hawaii = {}
}

if ( typeof(hawaii_sendMessage) == 'function' )
{
	hawaii.sendMessage = hawaii_sendMessage
}

if ( typeof(hawaii_registerMessageListener) == 'function' )
{
	hawaii.registerMessageListener = hawaii_registerMessageListener
}

_hawaii_hasStartedTick = false;

hawaii_interface = 
{
	sendMessage: function( obj )
	{
		hawaii.sendMessage( JSON.stringify( obj ) );
	},

	registerMessageListener: function( listenerFunction )
	{
		hawaii.registerMessageListener( listenerFunction );
	},

	_tick: function()
	{
		// note: this is not JSON, it is interpreted by the native interface & used to poll the native Hawaii reporting pipeline
		hawaii.sendMessage( "tick" );
	},

	startTick: function()
	{
		if ( !_hawaii_hasStartedTick )
		{
			_hawaii_hasStartedTick = true;
			setInterval( this._tick, 1000 / 10 );
		}
	},
}

muve.player.adapters.define("HawaiiAdapter",
{
	_abrConfig: null,
	_drmConfig: null,
	_hawaiiInterface: null,
	_rubyConfigJSON: null,
	_currentBandwidth: null,
	_currentTime: null,
	_currentUrl: null,
	_currentBitrate: null,
	_isSeeking: null,
	_isStartingPlayback: null,
	_isBuffering: null,
	_isStopping: null,
	_isPlaying: null,
	_duration: null,
	_outputWidth: null,
	_outputHeight: null,
	_droppedFrames: null,
	_availableBitrates: null,
	_hasReportedMetadataLoaded: null,
	_fragmentReportFrequency: null,
	_fragmentReports: null,

	_onMessage: function( msg )
	{
		var jsonMessage = JSON.parse( msg );

		switch ( jsonMessage.cmd )
		{
		case "playback_time":
			// convert from seconds to milliseconds
			this._currentTime = jsonMessage.params.playback_time * 1000.0;
			break;
		case "report_output":
			switch ( jsonMessage.params.type )
			{
			case "bandwidth":
				if ( jsonMessage.params.download_type === "video" )
				{
					this._currentBandwidth = jsonMessage.params.bandwidth;
				}	
				break;
			case "manifest_requested":
				this._sendMuveMessage( "player_connection_attempted" );
				break;
			case "manifest_obtained":
				this._sendMuveMessage( "player_connection_established" );
				break;
			case "dropped_frame":
				this._droppedFrames += 1;
				break;
			case "smooth_streaming":
				if ( !this._hasReportedMetadataLoaded )
				{
					if ( jsonMessage.params.download_type === "video" )
					{
						this._currentBitrate = jsonMessage.params.bitrate;

						var metadata = muve.player.media.create("Metadata");

						metadata.availableBitrates = this._availableBitrates;
						metadata.currentTime = this._currentTime;
						metadata.duration = this._duration;
						metadata.outputWidth = this._outputWidth;
						metadata.outputHeight = this._outputHeight;
						metadata.currentBitrate = this._currentBitrate;

						this._sendMuveMessageArgs( "player_metadata_loaded", metadata )
						this._hasReportedMetadataLoaded = true;
					}
				}

				this._fragmentReports.push( this._convertNativeToMuveFragmentData( jsonMessage.params ) );
				if ( this._fragmentReports.length >= this._fragmentReportFrequency )
				{
					this._fragmentReports.forEach( function(muveFragmentData)
					{
						this._sendMuveMessageArgs( "player_new_fragment_downloaded", muveFragmentData );
					}.bind(this));

					this._fragmentReports = []
				}
				break;
			case "video_quality_level":
				this._availableBitrates.push( jsonMessage.params.bitrate );
				break;
			case "metadata":
				// convert from seconds to milliseconds
				this._duration = jsonMessage.params.duration * 1000.0;
				break;
			default:
				break;
			}
			break;
		case "video_frame_quality_level_changed":
			this._currentBitrate = jsonMessage.params.new_bitrate;

			var newBitrateVal = jsonMessage.params.new_bitrate;
			var oldBitrateVal = jsonMessage.params.old_bitrate;
			this._sendMuveMessageArgs( "player_bitrate_switch_started", { targetedBitrate: newBitrateVal } );
			this._sendMuveMessageArgs( "player_bitrate_switch_completed", 
				{ newBitrate: newBitrateVal, oldBitrate: oldBitrateVal } );

			break;
		case "player_state_change":
			switch ( jsonMessage.params.state )
			{
			case "buffering":
				this._isBuffering = true;
				this._sendMuveMessage( "player_buffer_started" )
				break;
			case "playing":
				if (this._isSeeking)
				{
					this._isSeeking = false;
					this._sendMuveMessage( "player_seek_completed" )	
				}

				if (this._isBuffering)
				{
					this._sendMuveMessage( "player_buffer_completed" )
					this._isBuffering = false;
				}

				if (this._isStartingPlayback)
				{
					this._isStartingPlayback = false;
					this._sendMuveMessage( "player_ready" )
				}

				this._sendMuveMessageArgs( "player_playing", { startPosition: this.getCurrentTime() } )
				break;
			case "paused":
				this._sendMuveMessage( "player_paused" );
				break;
			case "stopped":
				if (this._isStartingPlayback)
				{
					// TODO ADD UNIT TESTING
					return;
				}

				this._isPlaying = false;

				if (this._isStopping)
				{
					this._sendMuveMessage( "player_stopped" );	
				}
				else
				{
					// native is notifying us that playback has ended
					this._sendMuveMessage( "player_ended" );
				}
				
				break;
			}
			break;
		case "network_disconnected":
			this._sendMuveMessage("player_network_error");
			break;
		case "network_connected":
			this._sendMuveMessage("player_network_connected");
			break;
		case "error":
			var errorCode = jsonMessage.params.code;
			var errorMessage = jsonMessage.params.message;
			this._raiseMuveErrorArgs( errorCode, errorMessage );
			break;		
		case "screen_dimensions":
			this._outputWidth = jsonMessage.params.width;
			this._outputHeight = jsonMessage.params.height;
			break;
		case "license_request":
			this._sendMuveMessage( "player_drm_license_requested" )

			var drmService = muve.services.get("DrmApiService");
			uriDecodedBase64EncodedRequest = decodeURIComponent( jsonMessage.params.request ) /// @todo: test this
			drmService.getPlayReadyLicense( atob( uriDecodedBase64EncodedRequest ), this._playReadyLicenseResponse.bind(this), this._playReadyLicenseError.bind(this));
			break;
		case "shutdown_complete":
			this._reset();
			this._sendMuveMessage( "player_unloaded" );
			break;
		default:
			break;
		}
	},

	_convertNativeToMuveFragmentData: function( nativeFragmentData )
	{
		var muveFragmentData = {
			stream : nativeFragmentData.download_type,
			fragmentIndex : nativeFragmentData.fragment_index,
			bitrate : nativeFragmentData.bitrate,
			downloadTime : nativeFragmentData.time_to_download,
			serverIp : this._intIpToStringIp( nativeFragmentData.ip_address ),
			chunkUrl: this._convertNativeUrlToMuveUrl(nativeFragmentData.fragment_uri),
			chunkSize : nativeFragmentData.fragment_size,
		}

		return muveFragmentData;
	},

	_intIpToStringIp: function (int)
	{
	    var part1 = int & 255;
	    var part2 = ((int >> 8) & 255);
	    var part3 = ((int >> 16) & 255);
	    var part4 = ((int >> 24) & 255);

	    return part4 + "." + part3 + "." + part2 + "." + part1;
	},

	_convertNativeUrlToMuveUrl: function (value)
	{
		value = value || "";
		var fragmentsIndex = value.indexOf("/Fragments");
		if (fragmentsIndex !== -1)
		{
			value = value.substr(0, fragmentsIndex);
		}
		return value;
	},

	_playReadyLicenseResponse: function( base64EncodedResponse )
	{
		this._sendMuveMessage( "player_drm_license_received" )

		this._hawaiiInterface.sendMessage( { cmd: "license_response", params: { response: base64EncodedResponse}});
	},

	_playReadyLicenseError: function()
	{
		this._raiseMuveError(muve.player.errorType.LICENCE_NOT_RECEIVED, "PlayReadyLicenceError" );
	},

	_sendMuveMessage: function( message )
	{
		muve.player.sendMessage( message );
	},

	_sendMuveMessageArgs: function( message, args )
	{
		muve.player.sendMessage( message, args );
	},

	_raiseMuveErrorArgs: function( errorCode, errorMessage )
	{
		var errorDescription = "code [" + errorCode + "] message [" + errorMessage + "]"
		this._raiseMuveError( muve.player.errorType.GENERIC, errorDescription );
	},

	_raiseMuveError: function( muveErrorType, descriptionText )
	{
		muve.player.raiseError(muveErrorType,
			{
				errorMessage: descriptionText
			});
	},

	_reset: function()
	{		
		this._abrConfig = null;
		this._drmConfig = null;
		this._rubyConfigJSON = {};
		this._currentBandwidth = 0;
		this._currentTime = 0;
		this._currentUrl = null;
		this._currentBitrate = 0;
		this._isSeeking = false;
		this._isStartingPlayback = false;
		this._isBuffering = false;
		this._isStopping = false;
		this._isPlaying = false;
		this._duration = 0;
		this._droppedFrames = 0;
		this._availableBitrates = [];
		this._hasReportedMetadataLoaded = false;
		this._fragmentReportFrequency = 1;
		this._fragmentReports = [];
	},

	init: function()
	{
		this._hawaiiInterface = hawaii_interface;
		this._outputWidth = 0;
		this._outputHeight = 0;
		this._reset();

		this._hawaiiInterface.registerMessageListener( this._onMessage.bind( this ) );

		this._hawaiiInterface.startTick();

		this._hawaiiInterface.sendMessage( { cmd : "screen_dimensions_request" } );
	},

	setPlayerConfig: function( playerConfig )
	{
		if ( playerConfig )
		{
			this._rubyConfigJSON = playerConfig.rubyConfigJSON;

			if ( playerConfig.fragmentsDownloadedEventFrequency )
			{
				this._fragmentReportFrequency = playerConfig.fragmentsDownloadedEventFrequency;
			}
		}
	},

	setAbrConfig: function( abrConfig )
	{
		this._abrConfig = abrConfig;
	},

	getAbrConfig: function()
	{
		return this._abrConfig;
	},

	setDrmConfig: function( drmConfig )
	{
		this._drmConfig = drmConfig;
	},

	getDrmConfig: function()
	{
		return this._drmConfig;
	},

	load: function( config )
	{
		if ( config )
		{
			if ( config.assets )
			{
				if ( config.assets.length > 0 )
				{
					var url = config.assets[0].url;
					if ( url )
					{
						this._hawaiiInterface.sendMessage( { cmd:"player_init", params:{ config : btoa( JSON.stringify( this._rubyConfigJSON ) ) } } );
						this._hawaiiInterface.sendMessage( { cmd:"player_control",params:{ call_function: "open", function_param: url } } );
						this._currentUrl = url;
						this._isStartingPlayback = true;
						this._sendMuveMessage( "player_setup_completed" )

						return;
					}
				}
			}
		}

		throw new Error("Missing asset uri in config");
	},

	getCurrentBandwidth: function()
	{
		return this._currentBandwidth;
	},

	_getProgressInfo: function()
	{
		
		var mediaProgressInfo = muve.player.media.create('ProgressInfo');

		mediaProgressInfo.currentTime = this.getCurrentTime();
		
		return mediaProgressInfo;
	},

	dispatchProgressInfo: function()
	{
		/// @brief We are constantly updating player progress, so notify muve that a value is available
		this._sendMuveMessageArgs( "player_progress_updated", this._getProgressInfo() );
	},

	getCurrentTime: function()
	{
		return this._currentTime;
	},

	getDuration: function()
	{
		return this._duration;
	},

	getAssetUrlForCurrentCdn: function()
	{
		return this._currentUrl;
	},

	play: function( startTimeMilliseconds )
	{
		this._isPlaying = true;

		if ( startTimeMilliseconds > 0 )
		{
			this.seek( startTimeMilliseconds );			
		}

		if (this._isSeeking)
		{
			this._sendMuveMessage( "player_seek_started" )
		}

		this._hawaiiInterface.sendMessage( {cmd:"player_control",params:{call_function:"play"}} );
	},

	getFrameRateAverage: function()
	{
		return 0;
	},

	getDroppedFrames: function()
	{
		return this._droppedFrames;
	},

	getOutputWidth: function()
	{
		return this._outputWidth;
	},

	getOutputHeight: function()
	{
		return this._outputHeight;
	},

	getAvailableBitrates: function()
	{
		return this._availableBitrates;
	},

	getCurrentBitrate: function()
	{
		return this._currentBitrate;
	},

	pause: function()
	{
		this._hawaiiInterface.sendMessage( {cmd:"player_control", params: {call_function:"pause"}} )
	},

	seek: function( seekTimeMilliseconds )
	{
		this._isSeeking = true;	
		this._currentTime = seekTimeMilliseconds;	
		this._hawaiiInterface.sendMessage( {cmd:"player_control",params:{call_function:"seek", function_param:seekTimeMilliseconds}} );
	},

	stop: function()
	{
		if (this._isPlaying)
		{
			this._isStopping = true;
			this._hawaiiInterface.sendMessage( {cmd:"player_control", params: {call_function:"stop"}} );
		}
		else
		{
			this._sendMuveMessage( "player_stopped" );
		}
	},

	unload: function()
	{
		this._hawaiiInterface.sendMessage( {cmd:"player_control", params: {call_function:"shutdown"}} );
	},
	
	getBandwidth: function(callback)
	{
		var defaultBandwidth = 123456;
		var response = { speedTest:{} };

		response.speedTest.finalSpeed = defaultBandwidth;
		response.speedTest.averageSpeed = defaultBandwidth;
		response.speedTest.standardDeviation = 0;
		response.speedTest.date = Date.now();

		callback(response);
	},

	runSpeedTest: function(url, callback)
	{
		var defaultBandwidth = 123456;
		var response = {};

		response.finalSpeed = defaultBandwidth;
		response.averageSpeed = defaultBandwidth;
		response.standardDeviation = 0;
		response.date = Date.now();

		callback(response);
	},

});