-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

--[[
-- Applies Http Request Extension Augmentations
--]]
return function ()
	local _ffi = require('ffi')
	local common = require('LuaHttpRequestBinding_common')
	
	-- Weak key table for tracking signal instance
	local weakKeyMetatable = { __mode = 'k' }
	local weakSignalTable = {}
	setmetatable(weakSignalTable, weakKeyMetatable)
	
	-- Create Global Namespace
	local ns = common.createNamespace(_G, common.httpNamespaceHeirarchy)
	assert(type(ns) == 'table', 'Namespace must be a table');
	
	-- Enumerations
	ns.method = require('LuaHttpRequestBinding_HttpMethod')
	ns.state = require('LuaHttpRequestBinding_HttpState')
	ns.requestDispatchMode = require('LuaHttpRequestBinding_HttpRequestDispatchMode')
	
	-- FFI Symbols
	ns.HttpRequest = common.getFFIUserDataSymbol('HttpRequestSharedPtr')
	ns.HttpResponse = common.getFFIUserDataSymbol('HttpResponseSharedPtr')
	ns.HttpHeaders = common.getFFIUserDataSymbol('HttpHeadersSharedPtr')
	ns.HttpHeader = common.getFFIUserDataSymbol('HttpHeader')
	ns.IHttpCacheProvider = common.getFFIUserDataSymbol('IHttpCacheProviderSharedPtr')
	ns.CacheServiceHttpCacheProvider = common.getFFIUserDataSymbol('CacheServiceHttpCacheProviderSharedPtr')
	ns.IHttpResponseConsumer = common.getFFIUserDataSymbol('IHttpResponseConsumerSharedPtr')
	ns.BufferHttpResponseConsumer = common.getFFIUserDataSymbol('BufferHttpResponseConsumerSharedPtr')
	ns.FileHttpResponseConsumer = common.getFFIUserDataSymbol('FileHttpResponseConsumerSharedPtr')
	ns.IHttpClient = common.getFFIUserDataSymbol('IHttpClientSharedPtr')
	
	-- Factory Lib Tables
	ns.request = _G.network_http_HttpRequestFactory
	ns.cache = _G.network_http_HttpCacheProviderFactory
	ns.consumer = _G.network_http_HttpResponseConsumerFactory
	
	-- Make string (const chat*) getters nicer to work with
	-- by making them return reguar lia strings
	common.augmentStringGetter('getUrl', 'HttpRequestSharedPtr')
	common.augmentStringGetter('getBody', 'HttpRequestSharedPtr')
	common.augmentStringGetter('getFirst', 'HttpHeader')
	common.augmentStringGetter('getSecond', 'HttpHeader')
	common.augmentStringGetter('getHeader', 'HttpHeadersSharedPtr')
	common.augmentStringGetter('getType', 'IHttpResponseConsumerSharedPtr')
	common.augmentStringGetter('getType', 'FileHttpResponseConsumerSharedPtr')
	common.augmentStringGetter('getPath', 'FileHttpResponseConsumerSharedPtr')
	common.augmentStringGetter('getType', 'BufferHttpResponseConsumerSharedPtr')
	
	-- Special augmentation for HttpHeaders:all
	-- Build a normal lua table from the FFI HttpHeader userdata returned
	local httpHeadersMeta = common.getMetatableFor('HttpHeadersSharedPtr')
	local httpHeaderMeta = common.getMetatableFor('HttpHeader')
	local httpHeadersMeta_all_internal = httpHeadersMeta.all
	function httpHeadersMeta:all()
		local size = self:size();
		local result = {}
		if size > 0 then
			local headers = _ffi.new('HttpHeaderUserData*[' .. size .. ']', {})
			local actual = httpHeadersMeta_all_internal(self, headers, size)
			for i = 0, actual - 1 do
				result[headers[i]:getFirst()] = headers[i]:getSecond()
				httpHeaderMeta.__gc(headers[i])
			end
		end
		return result
	end
	
	-- Special augmentation for BufferHttpResponseConsumer:asString
	-- Build a normal lua string from the response buffer
	local bufferHttpResponseConsumerMeta = 
		common.getMetatableFor('BufferHttpResponseConsumerSharedPtr')
	function bufferHttpResponseConsumerMeta:asString()
		local size = self:getSize()
		if size > 0 then
			return _ffi.string(self:asPtr(), size)
		end
	end
	
	-- Special augmentation for the HttpRequest signals. 
	-- We are unable to easily bind these directly to the user data so
	-- we must store them in a weak-key lookup table and create on demand.
	local function createSignalGetter(signalName, meta)
		meta[signalName] = function(self)
			local signalTable
			
			if self == nil then
				error('Missing `self`. Did you forget to use a colon?')
			end
			
			-- Create the instances signal table if it doen't exist.
			if weakSignalTable[self] == nil then
				weakSignalTable[self] = {}
			end
			signalTable = weakSignalTable[self]
			
			-- Create the instance's signal slot if it doen't exist.
			if signalTable[signalName] == nil then
				signalTable[signalName] = 
					network.http.request.__createNativeSignalBinding(
					self, signalName)
			end
			
			return signalTable[signalName]
		end
	end
	
	local httpRequestMeta = common.getMetatableFor('HttpRequestSharedPtr');
	
	-- BE AWARE
	-- The signal name constants defined here affect logic in the C++
	-- HttpRequestFactoryLuaLib. If these names change, the C++ logic must 
	-- be updated. Failure to do this will result in un-bound signals.
	-- See: ./internal/src/bindings/lua/lib/HttpRequestFactoryLuaLib.cpp
	
	createSignalGetter('complete', httpRequestMeta)
	createSignalGetter('progress', httpRequestMeta)
	createSignalGetter('stateChange', httpRequestMeta)
end