/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef PLUGIN_NETWORK_HTTP_HTTP_REQUEST_H_
#define PLUGIN_NETWORK_HTTP_HTTP_REQUEST_H_

#include <NetworkConfig.h>
#include <Atomic.h>
#include <SmartPointers.h>
#include <signals/Signal.h>
#include <utils/TimerUtil.h>
#include <core/thread/Condition.h>
#include <core/thread/Thread.h>

#include <http/HttpMethod.h>
#include <http/HttpHeaders.h>
#include <http/HttpState.h>
#include <http/HttpResponse.h>
#include <http/client/IHttpClient.h>
#include <http/cache/IHttpCacheProvider.h>
#include <sm/IBindable.h>
#include <timeline/Timeline.h>

namespace ignition
{
namespace network
{
namespace http
{

class HttpRequest;
typedef core::memory::SharedPtr<HttpRequest> HttpRequestPtr;
typedef core::memory::WeakPtr<HttpRequest> HttpRequestWeakPtr;

class PLUGINNETWORK_API HttpRequest :
	public core::memory::enable_shared_from_this<HttpRequest>,
	public javascript::sm::IBindable
{
	DECLARE_BINDABLE_ID(HttpRequest);

public:
	typedef uint32_t TimeoutDurationMs;

	// This should be kept in sync with their Lua and JavaScript equivalents
	enum RequestDispatchMode
	{
		// Should use this mode in normal application for performance 
		REQUEST_DISPATCH_ASYNCHRONOUS,

		// Can use this to make unit testing easier
		REQUEST_DISPATCH_SYNCHRONOUS
	};

	static HttpRequestPtr create();
	virtual ~HttpRequest();

	void setRequestDispatchMode(RequestDispatchMode mode);

	const std::string& getUrl() const;
	void setUrl(const std::string& url);

	HttpMethod::Type getMethod() const;
	void setMethod(HttpMethod::Type method);
	void setMethod(const std::string& method);

	HttpHeaders& getHeaders();
	const HttpHeaders& getHeaders() const;
	void setHeaders(const HttpHeaders& headers);

	TimeoutDurationMs getTimeout() const;
	void setTimeout(TimeoutDurationMs timeout);

	TimeoutDurationMs getConnectTimeout() const;
	void setConnectTimeout(TimeoutDurationMs timeout);

	const std::string& getBody() const;
	void setBody(const std::string& body);

	HttpState::Type getState() const;

	void setResponse(HttpResponsePtr response);
	HttpResponsePtr getResponse();

	void setClient(client::IHttpClientPtr client);
	void setCacheProvider(cache::IHttpCacheProviderPtr cache);

	void setProgressInterval(core::MilliTime_t interval);
	core::MilliTime_t getProgressInterval() const;

	virtual void send();
	virtual void send(client::IHttpClientPtr client);

	virtual void sendSync();
	virtual void sendSync(client::IHttpClientPtr client);

	virtual void abort();

	core::signals::Signal<void(
			HttpState::Type state)> stateChange;

	core::signals::Signal<void(
			size_t downTotal,
			size_t downCurrent,
			size_t upTotal,
			size_t upCurrent)> progress;

	typedef core::signals::Signal<void()> CompleteSignal;
	CompleteSignal complete;

protected:
	void _setState(HttpState::Type state);
	HttpRequest();

private:

	static void _setStateStatic(HttpRequestWeakPtr weakSelf,
			HttpState::Type state);
	void _makeRequest();
	static void _makeRequestStatic(HttpRequestWeakPtr weakSelf);

	void _bindClientCallbacks();
	void _onComplete(HttpState::Type state);

	static void _onClientDataProgressCallbackStatic(
			HttpRequestWeakPtr weakSelf,
			size_t downTotal,
			size_t downCurrent,
			size_t upTotal,
			size_t upCurrent);

	RequestDispatchMode _dispatchMode;
	std::string _url;
	std::string _body;
	HttpMethod::Type _method;
	core::atomic<HttpState::Type> _state;
	HttpHeaders _headers;
	TimeoutDurationMs _timeout;
	TimeoutDurationMs _connectTimeout;
	client::IHttpClientPtr _client;
	cache::IHttpCacheProviderPtr _cache;
	HttpResponsePtr _response;
	core::MilliTime_t _progressInterval;
	core::MilliTime_t _startTime;
	bool _yieldedFromCache;
	core::thread::Mutex _syncMutex;
	core::thread::Condition _syncCondition;
	core::ActionId _actionId;
};

} // namespace http
} // namespace network
} // namespace ignition

#endif // PLUGIN_NETWORK_HTTP_HTTP_REQUEST_H_
