/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef PLUGIN_NETWORK_CONNECTIVITY_CONNECTIVITY_STATE_SERVICE_H_
#define PLUGIN_NETWORK_CONNECTIVITY_CONNECTIVITY_STATE_SERVICE_H_

#include <NetworkConfig.h>
#include <SmartPointers.h>
#include <service/Service.h>
#include <signals/Signal.h>
#include <sm/SignallingBindable.h>

namespace ignition
{
namespace network
{
namespace connectivity
{

class ConnectivityStateService;
typedef core::memory::SharedPtr<ConnectivityStateService> ConnectivityStateServicePtr;

class PLUGINNETWORK_API ConnectivityStateService :
	public core::memory::enable_shared_from_this<ConnectivityStateService>,
	public javascript::sm::SignallingBindable,
	public core::Service
{
public:
	DECLARE_BINDABLE_ID(ConnectivityStateService);

	//! @brief BE AWARE! This enum is expressed in various script bindings.
	//! Any changes made here must be expressed elsewhere in the various
	//! script bindings.
	enum State
	{
		// Connection is in an unknown state waiting for an activity to be started.
		// A connectivity check is required before the state will change.
		//
		// There may be network and Internet connectivity. Wait for
		// distinct result.		
		PENDING_NO_ACTIVITY = 0,

		// Connection is in an unknown state waiting for a connection to succeed
		// or fail.
		//
		// There may be network and Internet connectivity. Wait for
		// distinct result.
		CONNECTING = 1,

		// Connection is connected.
		//
		// There is a network connection but Internet connectivity is not
		// guaranteed.
		CONNECTED = 2,
		
		// Connection is present but is restricted by the platform
		//
		// This can happen where certain platforms required a certain criteria
		// to be fullfilled by the users, such an account login or 
		// parental restriction on internet connectivity.
		//
		// There is no guarantee of successful network traffic.
		CONNECTED_RESTRICTED_PLATFORM = 3,

		// Connection is present but is restricted by an external gateway
		//
		// This can can happen on network where there is a gateway which 
		// requires a login - or some other user action.
		// For example, hotel and airport WIFI networks.
		//
		// There is no guarantee of successful network traffic.
		CONNECTED_RESTRICTED_GATEWAY = 4,

		// Connection is not connected.
		//
		// There is no network connection and distinctly no Internet connectivity.
		DISCONNECTED = 5
	};

	static ConnectivityStateServicePtr create();
	static ConnectivityStateServicePtr get();
	static void set(ConnectivityStateServicePtr value);

	ConnectivityStateService();
	~ConnectivityStateService();

	virtual bool start();
	virtual bool stop();
	virtual bool isRunning();

	void setState(State value);
	State getState() const;

	core::signals::Signal<void(State)> stateChanged;

private:
	static ConnectivityStateServicePtr _instance;

	State _state;
	bool _isRunning;
};

} // namespace connectivity
} // namespace network
} // namespace ignition

#endif // PLUGIN_NETWORK_CONNECTIVITY_CONNECTIVITY_STATE_SERVICE_H_
