/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef IGNITION_NETWORK_UTILS_SINGLE_CONNECTION_SERVER_H_
#define IGNITION_NETWORK_UTILS_SINGLE_CONNECTION_SERVER_H_

#include <core/thread/Mutex.h>
#include <core/thread/Thread.h>
#include <UniquePtr.h>
#include <Function.h>

#include <queue>
#include <string>
#include <vector>

#include <SocketCompatibility.h>
#include <netinet/in.h>
#include <sys/select.h>

struct sockaddr_in;

namespace ignition
{
namespace network
{

class SingleConnectionServer
{
public:
	typedef std::vector<uint8_t> Bytes;
	typedef core::function<void()> Callback;

	class IDataReceiver
	{
	public:
		virtual ~IDataReceiver() {}

		virtual void push(const uint8_t* buffer,
				size_t length) = 0;
	};

	struct ServerOptions
	{
	public:
		ServerOptions(uint16_t requestedPortParam = DEFAULT_LISTEN_PORT,
				bool trySequentialPortsIfInUseInParam = true,
				unsigned int numberOfPortsToTryInParam = 5,
				const Callback& connectCallbackParam = Callback(),
				const Callback& disconnectCallbackParam = Callback());

		uint16_t requestedPort;
		bool trySequentialPortsIfInUse;
		unsigned int numberOfPortsToTry;
		Callback connectCallback;
		Callback disconnectCallback;
	};

	SingleConnectionServer(IDataReceiver& receiver,
			const std::string& threadName,
			const ServerOptions& serverOptions = ServerOptions());
	~SingleConnectionServer();

	static const uint16_t DEFAULT_LISTEN_PORT;

	void start();
	void stop();

	// This can be on a different thread from accept to unblock it
	void unblockSocketCalls();

	void send(const uint8_t* buffer, size_t length);
	inline void send(const Bytes& data)
	{
		send(&data[0], data.size());
	}
	inline void send(const std::string& data)
	{
		send((const uint8_t*)&data[0], data.size());
	}

	uint16_t getListenPort() const;

	bool isAcceptingConnections();
private:
	struct SocketAndPortData
	{
		SocketAndPortData(int socketParam = -1, uint16_t portParam = 0) :
				socket(socketParam),
				port(portParam)
		{
		}
		int socket;
		uint16_t port;
	};

	class CommunicationThread : public core::thread::Thread
	{
	public:
		CommunicationThread(SingleConnectionServer::IDataReceiver& receiver,
				const std::string& threadName,
				int unblockerSocket,
				const ServerOptions& serverOptions = ServerOptions());

		void stop();
		void send(const uint8_t* buffer, size_t length);

		bool isAcceptingConnections();

		uint16_t getListenPort() const;
	private:
		virtual core::thread::ThreadResult run();

		void _prepareToAcceptConnections();

		SocketAndPortData _createListeningSocket();
		void _bindToPort(uint16_t requestedPort,
				SocketAndPortData& data,
				sockaddr_in& addr,
				bool trySequentialPortsIfInUse,
				unsigned int numberOfPortsToTry);
		void _setAcceptingConnections(bool value);

		bool _accept();
		void _shutdownListeningSocket();
		
		void _callConnectCallback();

		void _receive();
		bool _receiveSelect(uint8_t* buf, unsigned int bufLength);

		void _handleDisconnection();

		core::thread::Mutex _runningMutex;
		IDataReceiver& _receiver;
		int _unblockerSocket;

		const ServerOptions& _serverOptions;

		core::thread::Mutex _acceptingConnectionsMutex;
		bool _acceptingConnections;

		SocketAndPortData _listeningSocket;

		int _clientSocket;
		fd_set _readFds;
		int _maxfd;
		bool _running;
	};

	class ConnectionUnblocker
	{
	public:
		ConnectionUnblocker();
		~ConnectionUnblocker();
		void unblock();
		int getSocket();
	private:
		int _socket;
		sockaddr_in _unblockerAddr;
	};

	static const int LISTEN_BACKLOG_SIZE = 128;

	int _createUnblockerSocket();
	static void _throwError(const std::string& message, int socket, bool closeSocket);

	IDataReceiver& _receiver;

	core::memory::UniquePtr<CommunicationThread> _communicationThread;
	const std::string _threadName;
	ConnectionUnblocker _connectionUnblocker;

	const ServerOptions _serverOptions;
};

} // namespace network
} // namespace ignition

#endif // IGNITION_NETWORK_UTILS_SINGLE_CONNECTION_SERVER_H_
