-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

MetricsManager = {}

local Metric = require('Metric')
local MetricTemplate = require('MetricTemplate')
local MetricSamplesBuffer = require('MetricSamplesBuffer')

local MetricsManager = class(
	function(self, callback)
		if callback == nil then
			callback = function() end
		end

		self._metrics = {}
		self._metricTemplates = {}
		self._metricCount = 0
		self._callback = callback
		self._samplesBuffer = MetricSamplesBuffer.new()

		self._handleStartMessage =
			function(message)
				self:handleStartMessage(message)
			end

		self._handleStopMessage =
			function(message)
				self:handleStopMessage(message)
			end

		self._handleInstantaneousActionMessage =
			function(message)
				self:handleInstantaneousActionMessage(message)
			end

		handleStartMessageSignal:add(self._handleStartMessage)
		handleStopMessageSignal:add(self._handleStopMessage)
		handleInstantaneousActionMessageSignal:add(
				self._handleInstantaneousActionMessage)
	end)

function MetricsManager:addMetric(metricData)
	if self:_checkIfTemplate(metricData) then
		self:_addMetricTemplate(metricData)
		return
	end

	local metric = Metric(metricData)

	self._metrics[metric:getName()] = metric
	self._metricCount = self._metricCount + 1
end

function MetricsManager:getMetricCount()
	return self._metricCount
end

function MetricsManager:unregister()
	handleStartMessageSignal:remove(self._handleStartMessage)
	handleStopMessageSignal:remove(self._handleStopMessage)
	handleInstantaneousActionMessageSignal:remove(
			self._handleInstantaneousActionMessage)
end

function MetricsManager:handleStartMessage(message)
	self:_handleMessage(message, Metric.MessageTypes.START)	
end

function MetricsManager:handleStopMessage(message)
	self:_handleMessage(message, Metric.MessageTypes.STOP)
end

function MetricsManager:handleInstantaneousActionMessage(message)
	self:_handleMessage(message, Metric.MessageTypes.INSTANTANEOUS)
end

function MetricsManager:getMetricSamplesCount()
	return self._samplesBuffer:getMetricSamplesCount()
end

function MetricsManager:getMetricSamples(startIndex, endIndex)
	return self._samplesBuffer:getMetricSamples(startIndex, endIndex)
end

function MetricsManager:storeMetricSample(name, data)
	self._samplesBuffer:storeMetricSample(name, data)
end

function MetricsManager:_handleMessage(message, messageType)
	for metricTemplateName, metricTemplate in pairs(self._metricTemplates) do
		local metricData = metricTemplate:handleMessage(message, messageType)
		if metricData ~= nil then
			if self._metrics[metricData.name] == nil then
				self:addMetric(metricData)
			end
			break
		end
	end

	for metricName, metric in pairs(self._metrics) do
		metric:handleMessage(message, messageType)
	end
end

function MetricsManager:_addMetricTemplate(metricData)
	local metricTemplate = MetricTemplate(metricData)
	if metricTemplate then
		self._metricTemplates[metricTemplate:getName()] = metricTemplate
	end
end

function MetricsManager:_checkIfTemplate(metricData)
	if metricData.name == nil then
		return false
	end

	return string.match(metricData.name, "[%a%d._]*{}[%a%d._]*")
end

return MetricsManager