-- Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local MetricTemplate = class(
	function(self, data)
		if data == nil or
			data.name == nil or
			data.markers == nil then
			return nil
		end

		self._name = data.name
		self:_setupMarkers(data.markers)
		self._callback = data.callback
		self:_setDefaults()

		return instance
	end)

function MetricTemplate:getName()
	return self._name
end

function MetricTemplate:handleMessage(message, messageType)
	-- check if actionName is nil
	local messageActionName = message:getActionName()
	if messageActionName == nil then
		return nil
	end

	-- check if template has markers
	if self._markers == nil or table.getn(self._markers) < 1 then
		return nil
	end

	-- get actionName if match
	local actionName = nil
	for _, marker in pairs(self._markers) do
		actionName = self:_getActionNameIfMatch(
			messageActionName, messageType, marker.name, marker.type);
		if actionName then break end
	end

	if actionName == nil then
		return nil
	end

	-- generate metric data
	return self:_generateMetricData(actionName)
end


function MetricTemplate:_getActionNameIfMatch(name, messageType, templateName, templateType)
	if templateType ~= messageType then
		return nil
	end

	local prefix, postfix = string.match(templateName, "([%a%d._]*){}([%a%d._]*)")
	if prefix == nil and postfix == nil then
		return nil
	end

	-- if match returns matched string, otherwise returns nil
	return string.match(name, prefix.."([%a%d._]*)"..postfix)
end

function MetricTemplate:_generateMarkers(actionName)
	local newMarkers = {}
	for index, marker in pairs(self._markers) do
		local markerData = {}
		for key, value in pairs(marker) do
			if (key == "name") then
				markerData[key] = string.gsub(value, "{}", actionName)
			else
				markerData[key] = value
			end
		end
		newMarkers[index] = markerData
	end
	return newMarkers
end

function MetricTemplate:_generateMetricData(actionName)
	local metricData = {}
	metricData.name = string.gsub(self._name, "{}", actionName)
	metricData.callback = self._callback
	metricData.markers = self:_generateMarkers(actionName)
	return metricData
end

function MetricTemplate:_setDefaults()
	if self._callback == nil then
		self._callback = function() end
	end
end

function MetricTemplate:_setupMarkers(markers)
	self._markers = {}

	for index, marker in pairs(markers) do
		if marker.name ~= nil and marker.type ~= nil then
			local newMarker = {}
			for key, value in pairs(marker) do
				newMarker[key] = value
			end
			self._markers[index] = newMarker
		end
	end
end

return MetricTemplate
