-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

MetricSamplesBuffer = {}

MetricSamplesBuffer.DEFAULT_BUFFER_SIZE = 1024
MetricSamplesBuffer.DEFAULT_INDEX_LIMIT = math.pow(2, 32)

local MetricSamplesBuffer = class(function(self, bufferSize, indexLimit)
	self._samples = {}
	self._index = 1
	self._bufferSize = bufferSize or MetricSamplesBuffer.DEFAULT_BUFFER_SIZE
	self._indexLimit = indexLimit or MetricSamplesBuffer.DEFAULT_INDEX_LIMIT
end)

function MetricSamplesBuffer:getMetricSamplesCount()
	return self._index - 1
end

function MetricSamplesBuffer:getMetricSamples(startIndex, endIndex)
	local samples = {}
	local index = self._index

	startIndex = startIndex or 1
	endIndex = endIndex or 0

	local startIndexMin = index - self._bufferSize
	if startIndex < startIndexMin then
		log.warn("MetricSamplesBuffer.getMetricSamples: " ..
				"startIndex " .. startIndex .. " is too low " ..
				"(expected " .. startIndexMin .. ")")
		startIndex = startIndexMin
	end
	local endIndexMax = index - 1
	if endIndex > endIndexMax then
		log.warn("MetricSamplesBuffer.getMetricSamples: " ..
				"endIndex " .. endIndex .. " is too high " ..
				"(expected " .. endIndexMax .. ")")
		endIndex = endIndexMax
	end

	for i = startIndex, endIndex do
		local index = self:_getIndexWithinSampleLimit(i)
		local name = self._samples[index].name
		local data = self._samples[index].data

		if samples[name] then
			table.insert(samples[name], data)
		else
			samples[name] = { data }
		end
	end

	return samples
end

function MetricSamplesBuffer:storeMetricSample(name, data)
	local sample = { name = name, data = data }
	local index = self._index

	if index < self._indexLimit then
		self._samples[self:_getIndexWithinSampleLimit(index)] = sample
		self._index = index + 1
		if self._index == self._indexLimit then
			log.warn("MetricSamplesBuffer.storeMetricSample: " ..
					"Index limit of " .. self._indexLimit .. " reached")
		end
	end
end

function MetricSamplesBuffer:_getIndexWithinSampleLimit(index)
	return math.fmod(index - 1, self._bufferSize) + 1
end

return MetricSamplesBuffer
