-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ffi = require('ffi')

local Metric = class(
	function(self, data)
		if data == nil then
			data = {name=''}
		end

		self._name = data.name
		self._callback = data.callback
		self._allowConcurrentInstances = data.allowConcurrentInstances

		self._runInstances = {}
		self._runInstanceCount = 0
		
		if data.reset ~= nil then
			self._resetMarkerData = 
			{
				marker = data.reset
			}
		end

		self:_setupMarkers(data.markers)

		self:_setDefaults()

		self:_addRunInstance()
		
		return instance
	end)

Metric.MessageTypes =
	{
		START = 'start',
		STOP = 'stop',
		INSTANTANEOUS = 'instantaneous'
	}

function Metric:_setDefaults()
	if self._callback == nil then
		self._callback = function() end
	end

	if self._allowConcurrentInstances == nil then
		self._allowConcurrentInstances = false
	end
end

function Metric:getRunInstanceCount()
	return self._runInstanceCount
end

function Metric:_addRunInstance()
	local instanceId = self._runInstanceCount + 1

	local newInstance = 
		{
			timeData = {},
			metaData = {},
			matchedIds = {},
			instanceId = instanceId,
			markerIndex = 1
		}

	self._runInstances[instanceId] = newInstance
	self._runInstanceCount = self._runInstanceCount + 1
	return newInstance
end

function Metric:_setupMarkers(markers)
	if markers == nil then
		markers = {}
	end

	self._markerData = {}

	for index, marker in pairs(markers) do
		if marker.type == nil then
			marker.type = Metric.MessageTypes.STOP
		end

		local newMarkerData = {}
		newMarkerData.marker = marker

		local referenceData = self:_getReferenceDataForMarker(marker, markers)
		newMarkerData.referenceData = referenceData

		self._markerData[index] = newMarkerData
	end
end

function Metric:_getReferenceDataForMarker(marker, allMarkers)
	local referenceData = nil

	if marker.indexOfMarkerIdToMatch ~= nil then
		referenceData = 
		{
			target = allMarkers[marker.indexOfMarkerIdToMatch]
		}
	end

	return referenceData
end

function Metric:_removeInstance(instance)
	self._runInstances[instance.instanceId] = nil
	self._runInstanceCount = self._runInstanceCount - 1

	-- Always have one run ready to receive first marker
	if self._runInstanceCount == 0 then
		self:_addRunInstance()
	end	
end

function Metric:getName()
	return self._name
end

function Metric:getMarkerCount()
	return table.getn(self._markerData)
end

function Metric:getMarkerIndex()
	if self._runInstanceCount ~= 1 then
		error({message='getMarkerIndex() is only valid with single instance'})
	end
	
	return self:getMarkerIndexForInstanceId(1)
end

function Metric:getMarkerIndexForInstanceId(instanceId)
	local instance = self._runInstances[instanceId]

	return instance.markerIndex
end

function Metric:handleMessage(message, messageType)
	if self:getMarkerCount() == 0 then
		return
	end

	local shouldAddNewRunInstance = false

	for instanceId, instance in pairs(self._runInstances) do
		local shouldReset = false

		local nextMarkerData = self:_getNextMarkerDataForInstance(instance)

		if self:_messageMatchesMarkerData(message, 
			messageType, 
			nextMarkerData, 
			instance) then
				self:_messageMatchedMarkerData(message, messageType, nextMarkerData, instance)

				local oldMarkerIndex = instance.markerIndex

				instance.markerIndex = oldMarkerIndex + 1

				if self:_hasInstanceReachedLastMarker(instance) then
					self:_callCallback(instance)
					shouldReset = true
				elseif oldMarkerIndex == 1 and self._allowConcurrentInstances then
					-- if concurrent actions are allowed, and this is the first marker,
					-- and it wont't reset
					-- then add another instance to track future first markers
					shouldAddNewRunInstance = true
				end
		elseif self._resetMarkerData ~= nil and
				self:_messageMatchesMarkerData(message, messageType, self._resetMarkerData) then
			shouldReset = true
		end

		if shouldReset then
			self:_removeInstance(instance)
			shouldAddNewRunInstance = false
		end
	end	

	if shouldAddNewRunInstance then
		self:_addRunInstance()
	end
end

function Metric:_getNextMarkerDataForInstance(instance)
	return self._markerData[instance.markerIndex]
end

function Metric:_messageMatchesMarkerData(message, messageType, markerData, instance)
	if not markerData then
		return false
	end

	local marker = markerData.marker

	local typeAndNameMatch =
			self:_getActionNameFromMessage(message) == marker.name and 
			messageType == marker.type

	if not typeAndNameMatch then
		return false
	end

	local referenceData = markerData.referenceData
	if referenceData ~= nil and message.getActionId then

		local matchedId = instance.matchedIds[#instance.matchedIds]
		if matchedId ~= message:getActionId() then
			return false
		end
	end

	local markerMetadata = marker.metadata
	if markerMetadata ~= nil then
		local messageMetadata = message:getMetadata()

		for key, value in pairs(markerMetadata) do
			local stringKey = tostring(key)
			if not messageMetadata:has(stringKey) then
				return false
			end
			local messageValue = messageMetadata:get(stringKey)
			if messageValue == nil or messageValue:get() ~= value then
				return false
			end
		end
	end

	return true
end

function Metric:_getActionNameFromMessage(message)
	return ffi.string(message:getActionName())
end

function Metric:_messageMatchedMarkerData(message, messageType, markerData, instance)
	local timeDataInstance = self:_getTimeDataForMatchedMarker(message,
		messageType,
		markerData.marker)

	instance.timeData[#instance.timeData + 1] = timeDataInstance

	self:_getMetaDataForMatchedMarker(message, instance.metaData)

	local actionId = -1
	if message.getActionId then
		actionId = message:getActionId()
	end

	instance.matchedIds[#instance.matchedIds + 1] = actionId
end

function Metric:_getMetaDataForMatchedMarker(message, instanceMetadata)

	local messageMetadata = message:getMetadata()

	if messageMetadata ~= nil then
		local metadataKeys = messageMetadata:keys()

		while metadataKeys:size() > 0 do
			local metadataKey = metadataKeys:back():get()
			metadataKeys:pop()

			local messageValue = messageMetadata:get(metadataKey):get()
			instanceMetadata[metadataKey] = messageValue
		end
	end
end

function Metric:_getTimeDataForMatchedMarker(message, messageType, marker)
	local newTimeData = 
	{
		markerName = marker.name,
		markerType = messageType
	}

	if messageType == Metric.MessageTypes.START then
		newTimeData.time = message:getStartTime()
	elseif messageType == Metric.MessageTypes.STOP then
		newTimeData.time = message:getStopTime()
	elseif messageType == Metric.MessageTypes.INSTANTANEOUS then
		newTimeData.time = message:getTime()
	end

	return newTimeData
end

function Metric:_hasInstanceReachedLastMarker(instance)
	return instance.markerIndex == self:getMarkerCount() + 1
end

function Metric:_callCallback(instance)
	local firstMarker = instance.timeData[1]
	local lastMarker = instance.timeData[#instance.timeData]

	local report =
	{
		metricName = self._name,
		metadata = instance.metaData,
		markers = instance.timeData,
		totalTime = lastMarker.time - firstMarker.time
	}
	self._callback(report)
end

return Metric