/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef METRICS_METRICS_MANAGER_H_
#define METRICS_METRICS_MANAGER_H_

#include "metrics/MetricSamples.h"
#include <timeline/utils/ITimelineListener.h>
#include <extensions/NativeSignalExtension.h>
#include <core/thread/ThreadPool.h>
#include <service/Service.h>
#include <SmartPointers.h>
#include <timeline/Timeline.h>
#include <utils/TimerUtil.h>

namespace ignition
{
namespace core
{
class Timeline;
} // namespace core

namespace lua
{
class LuaState;
class NativeSignalExtension;
class ClassExtension;
class LoggerExtension;
} // namespace lua

namespace metrics
{

class MessageLuaExtension;

class MetricsManager : public core::ITimelineListener, public core::Service
{
public:
	static const std::string LUA_START_SIGNAL_NAME;
	static const std::string LUA_STOP_SIGNAL_NAME;
	static const std::string LUA_INSTANTANEOUS_ACTION_SIGNAL_NAME;
	static const std::string LUA_GET_METRIC_SAMPLES;
	static const std::string LUA_GET_METRIC_SAMPLES_COUNT;

	MetricsManager();
	virtual ~MetricsManager();

	void setup();
	void registerTimelineListener(core::Timeline& timeline);

	lua::LuaState& getLuaState() const;

	virtual void receiveStartMessage(const ITimelineListener::StartMessage& message);
	virtual void receiveStopMessage(const ITimelineListener::StopMessage& message);
	virtual void receiveInstantaneousActionMessage(
			const ITimelineListener::InstantaneousActionMessage& message);
	virtual void processMessages();

	virtual void getPublishedMetricSamples(MetricSamplesPollingState& request);
	core::MilliTime_t getIgnitionStartupTimestamp() const;

	// from core::Service
	virtual bool start();
	virtual bool stop();
	virtual bool isRunning();

	void flush();
private:
	void _processStartMessage(ITimelineListener::StartMessage* message);
	void _processStopMessage(ITimelineListener::StopMessage* message);
	void _processInstantaneousActionMessage(
			ITimelineListener::InstantaneousActionMessage* message);

	void _addLuaSearchPath();
	void _setupLuaExtensions();
	virtual std::string _getMetricsSetupScriptPath() const;

	template <typename T>
	void _dispatchFFIMessageToLua(lua::LuaFFIClassInstance<T*> instance,
			lua::NativeSignalExtension::SignalPtr& signal);

	core::Timeline* _timeline;
	lua::LuaState* _luaState;
	lua::LuaRegistryRef _metricsManagerRef;
	
	MessageLuaExtension* _messageExtension;
	lua::NativeSignalExtension* _nativeSignalExtension;

	lua::NativeSignalExtension::SignalPtr _startMessageSignal;
	lua::NativeSignalExtension::SignalPtr _stopMessageSignal;
	lua::NativeSignalExtension::SignalPtr _instantaneousActionMessageSignal;

	core::ThreadPool* _threadPool;
	core::MilliTime_t _ignitionStartupTimestamp;
};

template <typename T>
void MetricsManager::_dispatchFFIMessageToLua(lua::LuaFFIClassInstance<T*> instance,
			lua::NativeSignalExtension::SignalPtr& signal)
{
	_luaState->lock();
	signal->dispatch(instance);
	_luaState->unlock();
}

} // namespace metrics
} // namespace ignition

#endif // METRICS_METRICS_MANAGER_H_
