
--[[

    Purely decorative class designed for enabling the supplied subject instance to
    have any getter or setter methods referred to as though they were public properties
    rather than private properties with getters and setters. Enables, for instance:

        node.x = parent.x

    Rather than:

        node.setX(parent.getX())

    This is achieved via use of Lua's __index and __newindex metamethods. To use this
    decorator, instantiate it and pass the object you wish to decorate to this class'
    setSubject() method. Then, rather than accessing your object directly, use the
    decorator instance as though it were your object to access its properties.

    For example:

        -- Instantiate the decorator and set the subject it should decorate
        local decorator = PropertyAccessDecorator.new()
        decorator.setSubject(node)

        -- Access getX() using property access style syntax rather than the getter
        print('x is ' .. decorator.x)

  ]]--

local PropertyAccessDecorator = {}

local GETTER_PREFIX = 'get'
local SETTER_PREFIX = 'set'

function PropertyAccessDecorator.new(subject)
	local self = {}

	local _subject = subject
	local _keyCaches = 
	{
		[GETTER_PREFIX] = {},
		[SETTER_PREFIX] = {}
	}

	local function _capitalizeKeyAndAddPrefix(prefix, key)
		local keyCache = _keyCaches[prefix]
		local cachedKey = keyCache[key]
		
		if (cachedKey == nil) then
			local transformedKey = prefix .. key:sub(1, 1):upper() .. key:sub(2)
			keyCache[key] = transformedKey
			return transformedKey
		else
			return cachedKey
		end
	end

	local function _throwMissingMethodError(methodName)
		error('Subject does not have a method called ' .. methodName)
	end

	-- Note that there's no check for whether _subject is nil before trying to do
	-- the function lookup on it. I've omitted this deliberately, because property
	-- access is going to be done in performance critical sections of code. I think
	-- it's ok to omit it, because the PropertyAccessDecorator itself will only be
	-- used in one section of the code, and so we can be sure we call setSubject()
	-- before trying to use it.

	local function _callGetterOnSubject(key)
		local getterName = _capitalizeKeyAndAddPrefix(GETTER_PREFIX, key)
		local getterFunc = _subject[getterName]

		if (getterFunc ~= nil) then
			return getterFunc(_subject)
		else
			_throwMissingMethodError(getterName)
		end
	end

	local function _callSetterOnSubject(key, value)
		local setterName = _capitalizeKeyAndAddPrefix(SETTER_PREFIX, key)
		local setterFunc = _subject[setterName]

		if (setterFunc ~= nil) then
			return setterFunc(_subject, value)
		else
			_throwMissingMethodError(setterName)
		end
	end

	-- Table of methods belonging to PropertyAccessDecorator. For the most part this
	-- class just inflects the key supplied to __index or __newindex to turn it into
	-- a getter or setter and then calls the getter or setter on the subject, but if
	-- the key matches the name of anything in this table then it will be returned
	-- instead.
	local _publicMethods =
	{
		setSubject = function (self, subject)
			_subject = subject
		end
	}

	setmetatable(self,
	{
		__index = function (self, key)
			-- See the comment above re the _publicMethods table for an explanation
			-- of this step.
			local publicMethodBelongingToSelf = _publicMethods[key]

			if publicMethodBelongingToSelf ~= nil then
				return publicMethodBelongingToSelf
			else
				return _callGetterOnSubject(key)
			end
		end,

		__newindex = function(self, key, value)
			_callSetterOnSubject(key, value)
		end
	})

	return self
end

return PropertyAccessDecorator



