--[[
--
--    Creates a wrapper for the TimelineBinding provided, making it possible
--    to call them
--
--]]
local function _wrapTimeline(timelineBinding)

	local timeline = {}

	-- Starts an action with the supplied name, returning the started ActionId.
	function timeline.startAction(name)
		return timelineBinding:startAction(name)
	end

	-- Stops an action, either with or without an ActionId being specified. If
	-- the ActionId is not specified, actions with the same name must not overlap
	-- (i.e. you must always stop an action before starting it again).
	function timeline.stopAction(name, id)
		if id == nil then
			timelineBinding:stopActionWithoutId(name)
		else
			timelineBinding:stopActionWithId(name, id)
		end
	end

	-- Sends an instantaneous action, for events which don't have a start/stop
	function timeline.triggerInstantaneousAction(name)
		timelineBinding:triggerInstantaneousAction(name)
	end

	-- Resets the timeline, removing all action groups and resetting the start
	-- and end times of the timeline itself.
	function timeline.reset()
		timelineBinding:reset()
	end

	-- Prints the timeline to disk.
	function timeline.print()
		timelineBinding:print()
	end
	
	return timeline

end

--[[
--
--    Builds the global namespace for the timeline extension.
--
--    This function also returns a table containing protected management methods
--    for the extension.
--
--]]
return function(namespace)
	assert(type(namespace) == 'string', 'Namespace must be a string')
	assert(_G[namespace] == nil, 'Global timeline namespace must not already exist')

	local timeline = {}
	_G[namespace] = timeline

	-- Functions for the native extension to call in order to configure various
	-- aspects of the timeline namespace.
	return
	{
		-- Sets the Timeline instance in use by the app.
		setTimeline = function (timelineBinding)
			_G[namespace] = _wrapTimeline(timelineBinding)
		end
	}
end