-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved. --

-- SignalBinding -------------------------------------------------
------------------------------------------------------------------

--[[
   Object that represents a binding between a Signal and a listener function.
   <br />- <strong>self is an internal constructor and shouldn't be called by regular users.</strong>
   <br />- inspired by Joa Ebert AS3 SignalBinding and Robert Penner's Slot classes.
   @author Miller Medeiros
   @constructor
   @internal
   @name SignalBinding
   @param Signalend signal Reference to Signal object that listener is currently bound to.
   @param Function listener Handler function bound to the signal.
   @param boolean isOnce If binding should be executed just once.
   @param Object [listenerContext] Context on which listener will be executed (object that should represent the `self` localiable inside listener function).
   @param Number [priority] The priority level of the event listener. (default = 0).
  ]]--
local SignalBinding = class(function(self, signal, listener, isOnce, listenerContext, priority)

	--[[
	   Handler function bound to the signal.
	   @type Function
	   @private
	  ]]--
	self._listener = listener

	--[[
	   If binding should be executed just once.
	   @type boolean
	   @private
	  ]]--
	self._isOnce = isOnce

	--[[
	   Context on which listener will be executed (object that should represent the `self` localiable inside listener function).
	   @memberOf SignalBinding
	   @name context
	   @type Object|nil
	  ]]--
	self.context = listenerContext

	--[[
	   Reference to Signal object that listener is currently bound to.
	   @type Signal
	   @private
	  ]]--
	self._signal = signal

	--[[
	   Listener priority
	   @type Number
	   @private
	  ]]--
	self._priority = priority or 0

	--[[
	   If binding is active and should be executed.
	   @type boolean
	  ]]--
	self.active = true

	--[[
	   Default parameters passed to listener during `Signal.dispatch` and `SignalBinding.execute`. (curried parameters)
	   @type Array|nil
	  ]]--
	self.params = nil
end)

local function concat(tableA, startIndexA, tableB, startIndexB)
	local tableC = {}
	for n = startIndexA, #tableA do
		table.insert(tableC, tableA[n])
	end
	for n = startIndexB, #tableB do
		table.insert(tableC, tableB[n])
	end
	return tableC
end

--[[
   Call listener passing arbitrary parameters.
   <p>If binding was added using `Signal:addOnce()` it will be automatically removed from signal dispatch queue, self method is used internally for the signal dispatch.</p>
   @param Arrayend [paramsArr] Array of parameters that should be passed to the listener
   @return *end Value returned by the listener.
  ]]--
function SignalBinding:execute(paramsArr)
	local handlerReturn, params, paramsArrStartIndex
	if (self.active) and (self._listener ~= nil) then
		-- if we're calling with a specific context then we need to axe off the
		-- 'self' argument which will be at the start of the paramsArr
		paramsArrStartIndex = (self.context ~= nil) and 2 or 1
		params = self.params and concat(self.params, 1, paramsArr, paramsArrStartIndex) or paramsArr

		if self.context ~= nil then
			handlerReturn = self._listener(self.context, unpack(params, 1, params.n))
		else
			handlerReturn = self._listener(unpack(params, 1, params.n))
		end

		if (self._isOnce) then
			self:detach()
		end
	end
	return handlerReturn
end

--[[
   Detach binding from signal.
   - alias to: mySignal:remove(myBinding:getListener())
   @return Function|nilend Handler function bound to the signal or `nil` if binding was previously detached.
  ]]--
function SignalBinding:detach()
	return self:isBound() and self._signal:remove(self._listener, self.context) or nil
end

--[[
   @return boolean `true` if binding is still bound to the signal and have a listener.
  ]]--
function SignalBinding:isBound()
	return (self._signal ~= nil) and (self._listener ~= nil)
end

--[[
   @return boolean If SignalBinding will only be executed once.
  ]]--
function SignalBinding:isOnce()
	return self._isOnce
end

--[[
   @return Function Handler function bound to the signal.
  ]]--
function SignalBinding:getListener()
	return self._listener
end

--[[
   @return Signalend Signal that listener is currently bound to.
  ]]--
function SignalBinding:getSignal()
	return self._signal
end

--[[
   Delete instance properties
   @private
  ]]--
function SignalBinding:_destroy()
	self._signal = nil
	self._listener = nil
	self.context = nil
end

--[[
   @return stringend String representation of the object.
  ]]--
function SignalBinding:toString()
	return '[SignalBinding isOnce:' .. self._isOnce .. ', isBound:' .. self:isBound() .. ', active:' .. self.active .. ']'
end

return SignalBinding
