-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved. --

local SignalBinding = require('signals/SignalBinding')

-- Signal --------------------------------------------------------
--
-- Note: This is a direct port of JS Signals (more info about which 
-- can be found here: http://millermedeiros.github.io/js-signals/)
-- designed for use in Ignition. Apart from converting to Lua, I
-- haven't made any special effort to reformat the code or bring
-- it in line with our coding guidelines, so that anyone comparing
-- this code with the original JS code side by side will be able
-- to make direct comparisons.
--
-- The source was taken from commit 1a82bdd of the JS Signals git
-- repo. Accompanying tests were also taken, which can be found in
-- the pluginlua/tests/scripts/spec/signals folder.
--
-- The original JS sources are MIT as per the readme.md file at
-- https://github.com/millermedeiros/js-signals/blob/master/README.md,
-- and I've since added the Amazon copyright header to this port.
--
-- Any questions, please feel free to email pcowburn@amazon.com.
--
------------------------------------------------------------------


local function isFunctor(listener)
	return type(listener) == 'table' 
		and type(getmetatable(listener).__call) == 'function'
end

local function validateListener(listener, fnName)
	if not isFunctor(listener) and (type(listener) ~= 'function') then
		error( ('listener is a required param of {fn}() and should be a Function.'):gsub('{fn}', fnName) )
	end
end

local Signal

--[[
   Custom event broadcaster
   <br />- inspired by Robert Penner's AS3 Signals.
   @name Signal
   @author Miller Medeiros
   @constructor
  ]]--
Signal = class(function(self)
	--[[
	   @type Array.<SignalBinding>
	   @private
	  ]]--
	self._bindings = {}

	self._prevParams = nil

	--[[
	   Signals Version Number
	   @type String
	   @const
	  ]]--
	self.VERSION = '::VERSION_NUMBER::'

	--[[
	   If Signal should keep record of previously dispatched parameters and
	   automatically execute listener during `add()`/`addOnce()` if Signal was
	   already dispatched before.
	   @type boolean
	  ]]--
	self.memorize = false

	--[[
	   @type boolean
	   @private
	  ]]--
	self._shouldPropagate = true

	--[[
	   If Signal is active and should broadcast events.
	   <p><strong>IMPORTANT:</strong> Setting self property during a dispatch will only affect the next dispatch, if you want to stop the propagation of a signal use `halt()` instead.</p>
	   @type boolean
	  ]]--
	self.active = true

	-- enforce dispatch to aways work on same context (#47)
	self.dispatch = function(firstArg, ...)
		-- if we're being called as a free function rather than a method then
		-- we need to prepend the self argument onto the varargs array first
		if firstArg ~= self then
			return Signal.dispatch(self, firstArg, ...)
		else
			return Signal.dispatch(firstArg, ...)
		end
	end
end)

--[[
   @param Function listener
   @param boolean isOnce
   @param Object [listenerContext]
   @param Number [priority]
   @return SignalBinding
   @private
  ]]--
function Signal:_registerListener(listener, isOnce, listenerContext, priority)

	local prevIndex = self:_indexOfListener(listener, listenerContext)
	local binding

	if (prevIndex ~= -1) then
		binding = self._bindings[prevIndex]
		if (binding:isOnce() ~= isOnce) then
			error('You cannot add' .. (isOnce and '' or 'Once') .. '() then add' .. ((not isOnce) and '' or 'Once') .. '() the same listener without removing the relationship first.')
		end
	else 
		binding = SignalBinding.new(self, listener, isOnce, listenerContext, priority)
		self:_addBinding(binding)
	end

	if (self.memorize and self._prevParams) then
		binding:execute(self._prevParams)
	end

	return binding
end

local function aIsHigherPriorityThanB(a, b)
	return a._priority > b._priority
end

--[[
   @param SignalBinding binding
   @private
  ]]--
function Signal:_addBinding(binding)
	table.insert(self._bindings, binding)
	table.sort(self._bindings, aIsHigherPriorityThanB)
end

--[[
   @param Function listener
   @return Number
   @private
  ]]--
function Signal:_indexOfListener(listener, context)
	for n = #self._bindings, 1, -1 do
		local cur = self._bindings[n]
		if (cur._listener == listener) and (cur.context == context) then
			return n
		end
	end
	return -1
end

--[[
   Check if listener was attached to Signal.
   @param Function listener
   @param Object [context]
   @return boolean if Signal has the specified listener.
  ]]--
function Signal:has(listener, context)
	self:_logWarningIfUsedAfterDisposal()
	return self:_indexOfListener(listener, context) ~= -1
end

--[[
   Add a listener to the signal.
   @param Function listener Signal handler function.
   @param Object [listenerContext] Context on which listener will be executed (object that should represent the `self` localiable inside listener function).
   @param Number [priority] The priority level of the event listener. Listeners with higher priority will be executed before listeners with lower priority. Listeners with same priority level will be executed at the same order as they were added. (default = 0)
   @return SignalBinding An Object representing the binding between the Signal and listener.
  ]]--
function Signal:add(listener, listenerContext, priority)
	self:_logWarningIfUsedAfterDisposal()
	validateListener(listener, 'add')
	return self:_registerListener(listener, false, listenerContext, priority)
end

--[[
   Add listener to the signal that should be removed after first execution (will be executed only once).
   @param Function listener Signal handler function.
   @param Object [listenerContext] Context on which listener will be executed (object that should represent the `self` localiable inside listener function).
   @param Number [priority] The priority level of the event listener. Listeners with higher priority will be executed before listeners with lower priority. Listeners with same priority level will be executed at the same order as they were added. (default = 0)
   @return SignalBinding An Object representing the binding between the Signal and listener.
  ]]--
function Signal:addOnce(listener, listenerContext, priority)
	self:_logWarningIfUsedAfterDisposal()
	validateListener(listener, 'addOnce')
	return self:_registerListener(listener, true, listenerContext, priority)
end

--[[
   Remove a single listener from the dispatch queue.
   @param Function listener Handler function that should be removed.
   @param Object [context] Execution context (since you can add the same handler multiple times if executing in a different context).
   @return Function Listener handler function.
  ]]--
function Signal:remove(listener, context)
	self:_logWarningIfUsedAfterDisposal()
	validateListener(listener, 'remove')

	local i = self:_indexOfListener(listener, context)
	if (i ~= -1) then
		self._bindings[i]:_destroy() --no reason to a SignalBinding exist if it isn't attached to a signal
		table.remove(self._bindings, i)
	end
	return listener
end

--[[
   Remove all listeners from the Signal.
  ]]--
function Signal:removeAll()
	self:_logWarningIfUsedAfterDisposal()

	for n = #self._bindings, 1, -1 do
		self._bindings[n]:_destroy()
		table.remove(self._bindings, n)
	end
end

--[[
   @return Number Number of listeners attached to the Signal.
  ]]--
function Signal:getNumListeners()
	self:_logWarningIfUsedAfterDisposal()
	return #self._bindings
end

--[[
   Stop propagation of the event, blocking the dispatch to next listeners on the queue.
   <p><strong>IMPORTANT:</strong> should be called only during signal dispatch, calling it before/after dispatch won't affect signal broadcast.</p>
   @see Signal.disable
  ]]--
function Signal:halt()
	self:_logWarningIfUsedAfterDisposal()
	
	self._shouldPropagate = false
end

local function shallowCopy(tableA)
	local tableB = {}
	for n = #tableA, 1, -1 do
		table.insert(tableB, tableA[n])
	end
	return tableB
end

--[[
   Dispatch/Broadcast Signal to all listeners added to the queue.
   @param ...*end [params] Parameters that should be passed to each handler.
  ]]--
function Signal:dispatch(...)
	self:_logWarningIfUsedAfterDisposal()
	
	if (not self.active) then
		return
	end

	local paramsArr = {n=select('#', ...), ...} --convert varargs to a table whilst allowing for nil args
	local n = #self._bindings
	local bindings

	if (self.memorize) then
		self._prevParams = paramsArr
	end

	if (not n) then
		--should come after memorize
		return
	end

	bindings = shallowCopy(self._bindings) --clone array in case add/remove items during dispatch
	self._shouldPropagate = true --in case `halt` was called before dispatch or during the previous dispatch.

	--execute all callbacks until end of the list or until a callback returns `false` or stops propagation
	--reverse loop since listeners with higher priority will be added at the end of the list
	for n = n, 1, -1 do
		if (not bindings[n]) or (not self._shouldPropagate) or (bindings[n]:execute(paramsArr) == false) then
			break
		end
	end
end

--[[
   Forget memorized arguments.
   @see Signal.memorize
  ]]--
function Signal:forget()
	self:_logWarningIfUsedAfterDisposal()
	
	self._prevParams = nil
end

--[[
   Remove all bindings from signal and destroy any reference to external objects (destroy Signal object).
   <p><strong>IMPORTANT:</strong> calling any method on the signal instance after calling dispose will throw errors.</p>
  ]]--
function Signal:dispose()
	self:_logWarningIfUsedAfterDisposal()

	self:removeAll()
	self._bindings = {}
	self._prevParams = nil
	self._wasDisposed = true
end

function Signal:_logWarningIfUsedAfterDisposal()
	if self._wasDisposed and log ~= nil and log.warn ~= nil then
		log.warn('Signal was used after disposal')
	end
end

--[[
   @return stringend String representation of the object.
  ]]--
function Signal:toString()
	return '[Signal active:' .. self.active .. ' numListeners:' .. self:getNumListeners() .. ']'
end

return Signal