-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

--[[
-- Mixes-in generic `get` method to the ReflectableValue meta-table
-- This provides a type agnostic getter - the type stored by the 
-- ReflectableValue will be returned. No need to check for type before 
-- invoking the C getters.
-- 
-- Usage:
--   value:get()
--   value:set(value)
--
--]]
return function (metatableName)
	assert(type(metatableName) == 'string', 'metatableName must be of type string')

	local _ffi = require('ffi')
	local metatable = getMetatable(metatableName)

	if metatable == nil then
		error('Application of the extended getter/setters failed. ' ..
				'Unable to resolve the meta-table for ' .. metatableName)
	end

	if (type(metatable.get) == 'function') or  (type(metatable.set) == 'function') then
		error('Application of the extended getter/setters failed. ' ..
				'For the following metatable, get and/or set have been defined already ' ..
				metatableName)
	end

	-- Throws if the given value of reflectable is not user data
	local function _assetReflectable(reflectable)
		local typename = type(reflectable)
	
		assert(
			typename == 'userdata' or typename == 'cdata', 
			metatableName .. ' value must be passed as userdata or cdata'
		)
	end

	-- Type Agnostic Getter
	-- Usage: reflectable:get()
	function metatable:get()
		_assetReflectable(self)

		local valueType = self:type()

		if valueType == 'TYPE_BOOL' then
			return self:getBool()
		elseif valueType == 'TYPE_DOUBLE' then
			return self:getDouble()
		elseif valueType == 'TYPE_FLOAT' then
			return self:getFloat()
		elseif valueType == 'TYPE_INT' then
			return self:getInt()
		elseif valueType == 'TYPE_STR' then
			return _ffi.string(self:getString())
		elseif valueType == 'TYPE_UINT' then
			return self:getUint()
		elseif valueType == 'TYPE_VEC2' then
			return self:getVec2()
		elseif valueType == 'TYPE_VEC3' then
			return self:getVec3()
		elseif valueType == 'TYPE_VEC4' then
			return self:getVec4()
		elseif valueType == 'TYPE_RECT' then
			return self:getRect()
		elseif valueType == 'TYPE_CUBOID' then
			return self:getCuboid()
		elseif valueType == 'TYPE_NONE' then
			return nil
		else
			error(metatableName .. ':get failed. Unrecognised type ' .. valueType)
		end
	end

	-- Type Agnostic Setter
	-- Usage: reflectable:get(value)
	function metatable:set(value)
		_assetReflectable(self)

		local valueType = type(value)

		if value == nil then
			self:erase()
		elseif valueType == 'boolean' then
			self:setBool(value)
		elseif valueType == 'number' then
			self:setFloat(value)
		elseif valueType == 'string' then
			self:setString(value)
		else
			error(metatableName .. ':set non lua primitive types are not ' ..
					'currently supported. Please use explicit setters.')
		end
	end
end