-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ffi = require('ffi')
local PluginInterface = require('plugins.PluginInterface')

ffi.cdef[[
const char* lua_PluginsExtension_resolvePath(const char* pluginUri);
const char* lua_PluginsExtension_getStaticAssetsPath(const char* pluginId);
const char* lua_PluginsExtension_getUserVarPath(const char* pluginId);
const char* lua_PluginsExtension_getBinaryPath(const char* pluginId);
const char* lua_PluginsExtension_getTestDataPath(const char* pluginId);
]]

local plugins = {}

-- Contains copies of the PluginPathConvention constants, which are populated
-- from C++ when PluginsExtension is loaded.
plugins.convention = {}

plugins.__configureConventionConstants__ = function(constants)
	for key, value in pairs(constants) do
		plugins.convention[key] = value
	end
end

plugins.resolvePath = function(pluginUri)
	local stringPtr = ffi.C.lua_PluginsExtension_resolvePath(pluginUri)
	if stringPtr == nil then
		return nil
	else
		return ffi.string(stringPtr)
	end
end

plugins.getBinaryPath = function(pluginId)
	local stringPtr = ffi.C.lua_PluginsExtension_getBinaryPath(pluginId)
	return ffi.string(stringPtr)
end

plugins.getStaticAssetsPath = function(pluginId)
	local stringPtr = ffi.C.lua_PluginsExtension_getStaticAssetsPath(pluginId)
	return ffi.string(stringPtr)
end

plugins.getUserVarPath = function(pluginId)
	local stringPtr = ffi.C.lua_PluginsExtension_getUserVarPath(pluginId)
	return ffi.string(stringPtr)
end

plugins.getBinaryPath = function(pluginId)
	local stringPtr = ffi.C.lua_PluginsExtension_getBinaryPath(pluginId)
	return ffi.string(stringPtr)
end

plugins.getTestDataPath = function(pluginId)
	local stringPtr = ffi.C.lua_PluginsExtension_getTestDataPath(pluginId)
	return ffi.string(stringPtr)
end

local FRAMEWORK_PLUGIN_ID_PREFIX = 'com.amazon.ignition.framework.'

-- For convenience, the plugins extension provides access to each of the
-- Ignition framework plugins via a shorthand `plugins.pluginName` alias - for 
-- example, `com.amazon.ignition.framework.style` is available at plugins.style.
--
-- This is done through the use of an __index metamethod, which takes the
-- supplied key, converts it to a plugin ID and instantiates a new PluginInterface.
-- This is then cached so that further references to it return the cached instance.

local pluginAliasTable = {}

plugins.setPluginAlias = function(alias, pluginId)
	-- Unset any cached interfaces
	rawset(plugins, alias, nil)

	pluginAliasTable[alias] = pluginId
end

plugins.getPluginAlias = function(alias)
	return pluginAliasTable[alias]
end

setmetatable(plugins,
{
	__index = function(self, pluginName)
		local pluginId

		if not pluginAliasTable[pluginName] then
			pluginId = FRAMEWORK_PLUGIN_ID_PREFIX .. pluginName
		else
			pluginId = pluginAliasTable[pluginName]
		end

		local pluginInterface = PluginInterface.new(pluginId)
		
		-- Cache the PluginInterface instance against the global plugins object.
		rawset(plugins, pluginName, pluginInterface)
		
		return pluginInterface
	end
})

-- Public exports.
return plugins
