-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local LuaFileLoader = require('utils.LuaFileLoader')

-- This class contains a set of methods which provide access to the functions
-- exposed by PluginsExtension (getStaticAssetsPath, getUserVarPath etc), but 
-- wraps them so that they're scoped to an individual plugin.
--
-- For example, if instantiated with the plugin id 'com.amazon.ignition.framework.foo',
-- myPluginInterface:getStaticAssetsPath() will forward the call on to the generic
-- plugins.getStaticAssetsPath() function but automatically supply the pluginId
-- argument.
--
-- Additionally, it provides a means to require() or lazyRequire() Lua files
-- directly from the plugin's lua/scripts folder. This is useful because Lua's
-- default require() function results in a series of expensive fopen() calls
-- as it looks through each of the paths in the package.path list. Requiring 
-- files directly from the plugin means we can go straight to where we know
-- the file is on disk, and avoid the search step altogether.
local PluginInterface = class(function(self, pluginId)
	self._pluginId = pluginId
	self._pathCache = {}
end)

function PluginInterface:getPluginId()
	return self._pluginId
end

function PluginInterface:getStaticAssetsPath()
	return self:_retrieveAndCachePath('getStaticAssetsPath')
end

function PluginInterface:getUserVarPath()
	return self:_retrieveAndCachePath('getUserVarPath')
end

function PluginInterface:getBinaryPath()
	return self:_retrieveAndCachePath('getBinaryPath')
end

function PluginInterface:getTestDataPath()
	return self:_retrieveAndCachePath('getTestDataPath')
end

function PluginInterface:_retrieveAndCachePath(getterName)
	local cachedPath = self._pathCache[getterName]

	if cachedPath then
		return cachedPath
	else
		local path = plugins[getterName](self._pluginId)
		self._pathCache[getterName] = path
		return path
	end
end

-- Can be used in place of Lua's default require() function in order to avoid 
-- costly search path lookups. Loads the Lua file indicated by the supplied 
-- luaModuleFqn path, but does so by going straight to the corresponding plugin's
-- scripts/lua folder rather than scanning through package.path. For example,
-- the following call:
--
--     local Foo = myPlugin:require('some.module.Foo')
--
-- will result in path/to/myPlugin/scripts/lua/some/module/Foo.lua being loaded.
--
-- Additionally, the result of the require() call is cached so that subsequent 
-- calls do not result in unecessary disk access.
function PluginInterface:require(luaModuleFqn)
	-- When PluginInterface is being used by PluginsExtension to provide the 
	-- convenient plugins.style:require('bar.Baz') syntax, it's very easy for 
	-- the user to accidentally use a `.` rather than a `:` when calling require().
	--
	-- To avoid people wasting time trying to debug the rather cryptic error
	-- this produces, we print a more helpful error here.
	if type(self) ~= 'table' then
		error(
			'Missing `self` in PluginInterface:require(). Please ensure that ' ..
			'require is called with `:` notation rather than `.` notation.')
	end

	-- If the package is already loaded, or has been preloaded, we can bail early.
	-- This also helps to ensure reference equality with values returned by the 
	-- standard Lua require() function
	if package.loaded[luaModuleFqn] then
		return package.loaded[luaModuleFqn]
	elseif package.preload[luaModuleFqn] then
		return package.preload[luaModuleFqn]
	end

	-- Convert dots in the path to slashes, and append the .lua extension
	local luaModulePath = luaModuleFqn:gsub('%.', '/') .. '.lua'
	local result = self:_getOrCreateFileLoader():load(luaModulePath)
	
	-- Also place the result in package.loaded, so that any calls to
	-- the vanilla require() method will still return the same value.
	package.loaded[luaModuleFqn] = result
	
	return result
end

function PluginInterface:lazyRequire(luaModuleFqn)
	-- Passing a function as the second argument to the global lazyRequire()
	-- function causes it to use that function instead of require() when the
	-- module is loaded. Here we use configure it to use the plugin-specific 
	-- require method.
	return lazyRequire(luaModuleFqn, function(luaModuleFqn)
		return self:require(luaModuleFqn)
	end)
end

function PluginInterface:_getOrCreateFileLoader()
	if self._fileLoader == nil then
		local basePath = self:getStaticAssetsPath()
		local scriptsPath = plugins.convention.LUA_SCRIPTS_PATH
		self._fileLoader = LuaFileLoader.new(basePath .. scriptsPath)
	end
	
	return self._fileLoader
end

-- Just allows the file loader to be injected for testing.
function PluginInterface:setFileLoader(fileLoader)
	self._fileLoader = fileLoader
end

return PluginInterface

