local _type = type
local _pairs = pairs
local _ipairs = ipairs

local ensureLoaded = function(m)
	if _type(m) ~= "table" then
		return m
	end

	local mt = getmetatable(m)
	if mt == nil then
		return m
	end

	local loaded = mt["_loaded"]
	if loaded == false then
		loaded = mt["_requireFn"](mt["_modulename"])

		if _type(loaded) == "table" then
			for k,v in _pairs(loaded) do
				rawset(m, k, v)
			end
			setmetatable(m, getmetatable(loaded))
			return loaded
		elseif _type(loaded) == "function" then
			getmetatable(m).__call = function(m, ...)
				return loaded(...)
			end
		end

		mt["_loaded"] = loaded
	end

	return loaded or m
end

local enabledLazyRequireCall = function(m, module, requireFn)
	if package.loaded[module] ~= nil then
		return package.loaded[module]
	end

	local m = {}
	local mt =
	{

		__index = function(m, key)
			return ensureLoaded(m)[key]
		end,

		__newindex = function(m, key, value)
			rawset(ensureLoaded(m), key, value)
		end,

		__call = function(m, ...)
			return ensureLoaded(m)(...)
		end,

		_modulename = module,
		_loaded = false,
		_requireFn = requireFn or require
	}

	setmetatable(m, mt)
	return m
end

local disabledLazyRequireCall = function(m, module, requireFn)
	requireFn = requireFn or require
	return requireFn(module)
end

local lazyRequire =
{

	setEnabled = function(self, flag)
		if flag then
			type = function(m)
				return _type(ensureLoaded(m))
			end

			pairs = function(m)
				return _pairs(ensureLoaded(m))
			end

			ipairs = function(m)
				return _ipairs(ensureLoaded(m))
			end

			local mt = getmetatable(self)
			mt.__call = enabledLazyRequireCall
		else
			type = _type
			pairs = _pairs
			ipairs = _ipairs
			local mt = getmetatable(self)
			mt.__call = disabledLazyRequireCall
		end
	end,

	enabled = function(self)
		local mt = getmetatable(self)
		return mt.__call == enabledLazyRequireCall
	end
}

setmetatable(lazyRequire,
{
	__call = disabledLazyRequireCall
})

return lazyRequire
