
function init()

	local _ffi = require('ffi')
	local _libs = {}
	local _metatables = {}

	-- Loads a shared library using ffi.load
	local function _loadLib(libName)

		-- Load the lib and store it in the _libs table
		_libs[libName] = _ffi.load(libName)

	end

	-- Removes the supplied lib from the list of loaded libs
	local function _unloadLib(libName)

		-- FFI doesn't actually seem to provide a way to unload libs that have
		-- have been loaded via ffi.load, so we just do our best to clean up by
		-- nilling the reference we created.
		_libs[libName] = nil

	end

	-- Returns a reference to a lib loaded earlier by the supplied path
	local function _getLib(libName)

		return _libs[libName]

	end

	-- Builds up some Lua patterns for finding the /* @begin_ffi_externs */
	-- and /* @end_ffi_externs */ tags which delimit externs in the header file
	local _commentPrefix = '/%*%s*'
	local _commentSuffix = '%s*%*/'
	local _beginTag = '@begin_ffi_externs'
	local _endTag = '@end_ffi_externs'
	local _beginComment = _commentPrefix .. _beginTag .. _commentSuffix
	local _endComment = _commentPrefix .. _endTag .. _commentSuffix
	local _exportTag = '(PLUGIN)(%g*)(_API)'

	-- Loads in the supplied header file and extracts the externed functions
	local function _loadExternsAndAddCdef(libName, typeOrClassName, headerContents)

		local beginIndex = headerContents:find(_beginComment)
		local endIndex = headerContents:find(_endComment)

		if beginIndex == nil then
			error('Could not find /* ' .. _beginTag .. ' */')
		elseif endIndex == nil then
			error('Could not find /* ' .. _endTag .. ' */')
		end

		local externs = headerContents:sub(beginIndex, endIndex - 1)

		-- Remove compiler export tags 'PLUGIN*_API'
		externs = externs:gsub(_exportTag, ' ')

		local status, err = pcall(function()
			_ffi.cdef(externs)
		end)

		if not status then
			print('Error parsing FFI header file ' .. typeOrClassName .. ' in ' .. libName .. ':')
			print()
			print('    ' .. err)
			print()
		end

		return externs

	end

	local NEW_METHOD_NAME = 'new'
	local GC_METHOD_NAME = 'gc'
	local EQ_METHOD_NAME = 'eq'

	-- Loops over each function in the list of externs that begins with the prefix
	-- we're looking for, and creates a metatable with a method for each extern
	local function _mapMethodsToExternedFunctions(typeName, lib, externs, prefix)

		local metatable = {}
		metatable.__index = metatable

		-- If the class has no methods we might not be provided with a prefix,
		-- hence this check
		if (prefix ~= nil) and (prefix ~= "") then
			local methodPattern = '([_%w][_%w%s]*%*?)%s+'..prefix .. '([^%(]*)'
			local foundMatches = false
			local funcsThatNeedGc = {}

			for fnRetType, fnName in externs:gmatch(methodPattern) do
				foundMatches = true

				local fnPointer = lib[prefix .. fnName]
				if fnRetType:sub(#fnRetType) == '*' then
					local fnPointeeType = fnRetType:gsub('%s', ' '):gsub('*', '')
					if fnPointeeType:find("UserData$") ~= nil then
						funcsThatNeedGc[#funcsThatNeedGc + 1] =
						{
							fnPointeeType = fnPointeeType,
							fnName = fnName,
							fnPointer = fnPointer
						}
						fnPointer = nil
					end
				end

				if fnPointer ~= nil then
					if fnName == GC_METHOD_NAME then
						metatable.__gc = fnPointer
					elseif fnName == EQ_METHOD_NAME then
						metatable.__eq = fnPointer
					else
						metatable[fnName] = fnPointer
					end
				end
			end

			if foundMatches == false then
				error("Could not find any externs with the prefix '" .. prefix .. "'")
			end

			for _, fnTable in pairs(funcsThatNeedGc) do
				local returnTypeMetatable = nil
				if fnTable.fnPointeeType == typeName then
					returnTypeMetatable = metatable
				else
					returnTypeMetatable = _metatables[fnTable.fnPointeeType]
				end

				if returnTypeMetatable == nil or returnTypeMetatable.__gc == nil then
					error("Method '"..fnTable.fnName.."' returns '"
							..fnTable.fnPointeeType.."*', but '"..fnTable.fnPointeeType
							.."' has no 'gc' method.")
				end

				metatable[fnTable.fnName] = function(...)
					return _ffi.gc(fnTable.fnPointer(...), returnTypeMetatable.__gc)
				end
			end
		end

		return metatable

	end

	local USER_DATA_SUFFIX = 'UserData'

	-- Adds the 'UserData' suffix to the supplied class name. This is done
	-- because class instances are not handled directly in Lua as pointers
	-- to the instance, but are instead wrapped in a LuaFFIUserData wrapper
	-- so that the instance can be associated back to the Lib and Lua State
	-- to which it belongs. See LuaFFIUserData.h and the CREATE_FFI_USER_DATA()
	-- macro for more info.
	local function _addUserDataSuffix(className)

		return className .. USER_DATA_SUFFIX

	end

	-- When a class is bound which inherits from a previously defined class,
	-- this method is used to retrieve the base class metatable so that it
	-- can be combined with that of the derived class
	local function _loadBaseClassMetatable(baseClassName)

		local metatable

		if (baseClassName ~= nil) and (baseClassName ~= "") then
			metatable = _metatables[_addUserDataSuffix(baseClassName)]

			if metatable == nil then
				error("Could not find a base class metatable called " .. baseClassName)
			end
		end

		return metatable

	end

	-- When a class is bound which inherits from a previously defined class,
	-- the metatables must be combined so that any methods belonging to the
	-- base class are available in the derived class metatable.

	-- This method creates a new table containing first the methods from the
	-- base class, and then and derived methods. If name clashes occur (i.e.
	-- if a method in the derived class overrides one from the base class) then
	-- the derived class methods are favoured.
	local function _mergeMetatables(baseMetatable, derivedMetatable)

		local combinedMetatable = {}

		for i, v in pairs(baseMetatable) do
			combinedMetatable[i] = v
		end

		for i, v in pairs(derivedMetatable) do
			combinedMetatable[i] = v
		end

		combinedMetatable.__index = combinedMetatable

		return combinedMetatable

	end

	-- Adds the supplied metatable to the global namespace and internal list
	-- of metatables.
	local function _addMetatable(name, metatable)

		_G[name] = _ffi.metatype(name, metatable)
		_metatables[name] = metatable

	end

	-- Removes the supplied metatable from the global namespace and internal
	-- list of metatables.
	local function _removeMetatable(name)

		if _metatables[name] == nil then
			error("Could not find a metatable with name " .. name .. " to unbind")
		else
			_G[name] = nil
			_metatables[name] = nil
		end

	end

	-- Creates a binding for the supplied type name, by reading the struct def
	-- from the supplied extern file.
	local function _bindType(
			libName,
			typeName,
			headerContents)

		_loadExternsAndAddCdef(libName, typeName, headerContents)
		_addMetatable(typeName, {})

	end

	-- Removes the binding for the supplied type name
	local function _unbindType(typeName)

		_removeMetatable(typeName)

	end

	-- Creates a binding for the supplied class name, by reading the externs
	-- and extracting all function names which map to methods of the class.
	--
	-- These are then matched up with the function pointers from the FFI lib,
	-- so that methods from the class' metatable in turn call those functions.
	local function _bindClass(
			libName,
			className,
			headerContents,
			externFnPrefix,
			baseClassName)

		local lib = _getLib(libName)
		local externs = _loadExternsAndAddCdef(libName, className, headerContents)
		local classMetatable = _mapMethodsToExternedFunctions(
				_addUserDataSuffix(className), lib, externs, externFnPrefix)
		local baseMetatable  = _loadBaseClassMetatable(baseClassName)

		-- If we're doing inheritance, merge the base class and derived class
		-- together so that base class methods are shared to the derived class
		if baseMetatable ~= nil then
			classMetatable = _mergeMetatables(baseMetatable, classMetatable)
		end

		_addMetatable(_addUserDataSuffix(className), classMetatable)

	end

	-- Removes the binding for the supplied class name
	local function _unbindClass(className)

		_removeMetatable(_addUserDataSuffix(className))

	end

	-- Returns the metatable associated with the supplied class name
	local function _getMetatable(className)

		local classMetatable = _metatables[_addUserDataSuffix(className)]
		local typeMetatable = _metatables[className]

		if classMetatable == nil and typeMetatable == null then
			error("No metatable exists with the name " .. className)
		else
			return classMetatable or typeMetatable
		end

	end

	-- Exports
	return
	{
		loadLib      = _loadLib,
		unloadLib    = _unloadLib,
		getLib       = _getLib,
		bindType     = _bindType,
		unbindType   = _unbindType,
		bindClass    = _bindClass,
		unbindClass  = _unbindClass,
		getMetatable = _getMetatable
	}

end

-- So that multiple instances of the ffiLibWrapper aren't evaluated and
-- exposed over the global name-space we guard with a global flag
if __ffiLibWrapperInitialised ~= true then
	local methods = init()
	__ffiLibWrapperInitialised = true

	loadLib      = methods.loadLib
	unloadLib    = methods.unloadLib
	getLib       = methods.getLib
	bindType     = methods.bindType
	unbindType   = methods.unbindType
	bindClass    = methods.bindClass
	unbindClass  = methods.unbindClass
	getMetatable = methods.getMetatable
end
