--[[

	Allows command line arguments to be passed through to Lua states.
	
	Usage from the command line:
	
	   ignition --lua-args="--myString=foo --myFloat=3.142 --myBool=true --myFlag"
	
	Each of these args will be exposed via commandLine.args.myVariableName, e.g.
	
	   commandLine.args.myString;   -- "foo"
	   commandLine.args.myFloat;    -- 3.142
	   commandLine.args.myBool;     -- true
	   commandLine.args.myFlag;     -- true
	
	The types are intelligently converted into corresponding Lua types based on
	inferring what the intended type is meant to be (things that look like booleans
	get converted to real booleans, things that look like numbers get converted to
	real numbers etc). If you pass an arg with no value (like myFlag above), it'll
	get converted to a boolean with a value of true.
	
]]--

local args = {}
local parseArgs
local readArg
local advanceUntil
local interpretStringValue

function parseArgs(argsString)
	local i = 1

	while i < #argsString do
		if argsString:sub(i, i) == '-' then
			i = readArg(argsString, i)
		end
		
		i = i + 1
	end
end

function readArg(argsString, i)
	-- Remove the - or -- from the start of the string
	local nameStartIndex = advanceUntil(argsString, '[^-]', i)

	-- Read the arg name by reading until the next = or space character
	local nameEndIndex = advanceUntil(argsString, '[= ]', nameStartIndex)
	local name = argsString:sub(nameStartIndex, nameEndIndex - 1)

	-- If the name is followed by an = sign, read the value
	if (argsString:sub(nameEndIndex, nameEndIndex) == '=') then
		-- Read until the next space character or the end of the string
		local valueStartIndex = nameEndIndex + 1
		local valueEndIndex = advanceUntil(argsString, ' ', valueStartIndex)
		local value = argsString:sub(valueStartIndex, valueEndIndex - 1)

		value = interpretStringValue(value)
		args[name] = value

		return valueEndIndex
	-- Otherwise, interpret it as a flag arg
	else
		args[name] = true

		return nameEndIndex
	end
end

function advanceUntil(string, pattern, startIndex)
	local index = string:find(pattern, startIndex)

	-- Return the index of the match if found
	if index ~= nil then
		return index
	-- Otherwise just return the end of the string
	else
		return #string + 1
	end
end

-- Analyses whether the supplied string appears to be a boolean or a number,
-- and if so returns it as one of those types. Otherwise, leaves it as a string.
function interpretStringValue(value)
	if value == 'true' then
		return true
	elseif value == 'false' then
		return false
	elseif tonumber(value) ~= nil then
		return tonumber(value)
	else
		return value
	end
end

-- Public exports.
return
{
	args = args,
	parseArgs = parseArgs
}