-- Helper function
local function isArray(table)
	local i = 0
	for _ in pairs(table) do
		i = i + 1
		if table[i] == nil then return false end
	end
	return true
end

--[[
-- Method for providing basic inheritance between lua "classes".
--
-- Usage with no constructor or inheritance:
--
--   MyClass = class()
--
-- Usage with a inheritance, no constructor:
--
--   MyClass = class(MyParentClass)
--
-- Usage with a constructor, no inheritance:
--
--   MyClass = class(function(self)
--     self.prop = "Hello"
--   end)
--
-- Usage with inheritance and a constructor:
--
--   MyClass = class(MyParentClass, function(self)
--     self.prop = "Hello"
--   end)
--
-- Instantiation. The following have exactly the same effect:
--
--   local instA = MyClass()
--   local instB = MyClass.new()
--
-- Instantiation - Passing arguments
--
--   MyClass = class(function(self, firstArg, secondArg)
--     ...
--   end)
--
--   local instA = MyClass("Hello", "Bye")
--
--]]
local class = function(parentClass, _ctor)
	local newClass = {}

	-- If the constructor arg is missing and the parentClass arg is
	-- a function, assume that the parentClass is actually a _ctor.
	if not _ctor and type(parentClass) == 'function' then
		_ctor = parentClass
		parentClass = nil

	-- If the first argument is a table, assume its a Class Definition and
	-- copy its members to the new class table
	elseif type(parentClass) == 'table' then
		for key,value in pairs(parentClass) do
			newClass[key] = value
		end

		newClass._parentClass = parentClass
	end

	-- Set-up the definition table as its own index
	newClass.__index = newClass

	local newClassMeta = {}

	-- If no constructor function has been supplied, create one which by default
	-- will pass through to the parent constructor
	if (_ctor == nil) and (parentClass ~= nil) and (parentClass._ctor ~= nil) then
		_ctor = function(instance, ...)
			parentClass._ctor(instance, ...)
		end
	end

	--[[
	-- Class constructor
	-- First 'self' argument is always ignored
	--]]
	newClass.new = function(...)
		local instance = {}

		setmetatable(instance,newClass)

		-- Run the constructor if present
		if _ctor then
			_ctor(instance,...)
		end

		return instance
	end

	-- Class constructor alias
	newClassMeta.__call = function(ignoredContextArg, ...)
		return newClass.new(...)
	end

	-- Class local reference to the constructor argument if present
	-- This allows a class's constructor to be called from decedent types
	newClass._ctor = _ctor

	--[[
	-- Method to check if the 'self' instance is in fact an instance of the
	-- given class type passed in.
	--]]
	newClass.instanceOf = function(self, referenceClass)
		local meta = getmetatable(self)

		-- Look up the inheritance chain to see if there is a match
		while meta do
			if meta == referenceClass then return true end
			meta = meta._parentClass
		end

		return false
	end

	--[[
	-- Returns a reference to the instance's class
	--]]
	newClass.getClass = function()
		return newClass
	end

	--[[
	-- Returns a reference to the instance's parent class
	--
	-- USE WITH CAUTION! Calling this method from within a member
	-- method of a decendeent will not necessarily give you access
	-- to a super method you're looking for. This method will return
	-- the class type of the parent of the class instance describe
	-- by self.
	--
	-- For example, given the following inheritance chain:
	--
	--   ClassA -> ClassB -> ClassC
	--
	-- If you have an instance of ClassC and you call a member
	-- method on ClassB. If member method contains a call to
	-- getParentClass(), you will not get back ClassA.
	-- You will get back ClassB.
	--
	-- Where you need access to a method on a specific ancestor,
	-- prefer calling it explicitly by class instance, like:
	--
	--   ClassA.myMethod(self, ...)
	--
	--]]
	newClass.getParentClass = function()
		return parentClass
	end

	-- Receives either a single or multiple mixins, e.g.
	--
	-- local MixinA = { foo = function() end }
	-- local MixinB = { bar = function() end }
	--
	-- e.g. single mixin:     MyClass:mixin(MixinA)
	-- e.g. multiple mixins:   MyClass:mixin(MixinA, MixinB)
	newClass.mixin = function(self, ...)
		for _, mixin in ipairs({...}) do
			for propName, propValue in pairs(mixin) do
				if propName ~= "_ctor" then
					self[propName] = propValue
				end
			end
		end
	end

	-- Associate the new metatable with the new class
	setmetatable(newClass, newClassMeta)

	if mixins and isArray(mixins) then
		newClass:mixin(unpack(mixins))
	elseif mixins then
		newClass:mixin(mixins)
	end

	return newClass
end

return class
